/*
 * Decompiled with CFR 0.152.
 */
package com.google.api.auth;

import com.google.api.auth.AuthTokenDecoder;
import com.google.api.auth.CachingAuthTokenDecoder;
import com.google.api.auth.CachingJwksSupplier;
import com.google.api.auth.DefaultAuthTokenDecoder;
import com.google.api.auth.DefaultAuthTokenVerifier;
import com.google.api.auth.DefaultJwksSupplier;
import com.google.api.auth.DefaultKeyUriSupplier;
import com.google.api.auth.IssuerKeyUrlConfig;
import com.google.api.auth.UnauthenticatedException;
import com.google.api.auth.UserInfo;
import com.google.api.control.model.MethodRegistry;
import endpoints.repackaged.com.google.api.AuthProvider;
import endpoints.repackaged.com.google.api.Authentication;
import endpoints.repackaged.com.google.api.Service;
import endpoints.repackaged.com.google.api.client.http.GenericUrl;
import endpoints.repackaged.com.google.api.client.http.HttpRequestFactory;
import endpoints.repackaged.com.google.api.client.http.javanet.NetHttpTransport;
import endpoints.repackaged.com.google.api.client.util.Clock;
import endpoints.repackaged.com.google.api.client.util.Maps;
import endpoints.repackaged.com.google.api.config.ServiceConfigFetcher;
import endpoints.repackaged.com.google.common.annotations.VisibleForTesting;
import endpoints.repackaged.com.google.common.base.Optional;
import endpoints.repackaged.com.google.common.base.Preconditions;
import endpoints.repackaged.com.google.common.base.Strings;
import endpoints.repackaged.com.google.common.collect.ImmutableMap;
import endpoints.repackaged.com.google.common.collect.Sets;
import endpoints.repackaged.com.google.common.flogger.FluentLogger;
import endpoints.repackaged.org.jose4j.jwt.JwtClaims;
import endpoints.repackaged.org.jose4j.jwt.MalformedClaimException;
import endpoints.repackaged.org.jose4j.jwt.NumericDate;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.servlet.http.HttpServletRequest;

public class Authenticator {
    private static final FluentLogger logger = FluentLogger.forEnclosingClass();
    private static final String ACCESS_TOKEN_PARAM_NAME = "access_token";
    private static final String BEARER_TOKEN_PREFIX = "Bearer ";
    private static final String EMAIL_CLAIM_NAME = "email";
    private final AuthTokenDecoder authTokenDecoder;
    private final Clock clock;
    private final Map<String, String> issuersToProviderIds;

    @VisibleForTesting
    Authenticator(AuthTokenDecoder authTokenDecoder, Clock clock, Map<String, String> issuersToProviderIds) {
        this.authTokenDecoder = authTokenDecoder;
        this.clock = clock;
        this.issuersToProviderIds = issuersToProviderIds;
    }

    public UserInfo authenticate(HttpServletRequest httpServletRequest, MethodRegistry.AuthInfo authInfo, String serviceName) {
        Preconditions.checkNotNull(httpServletRequest);
        Preconditions.checkNotNull(authInfo);
        Optional<String> maybeAuthToken = Authenticator.extractAuthToken(httpServletRequest);
        if (!maybeAuthToken.isPresent()) {
            throw new UnauthenticatedException("No auth token is contained in the HTTP request");
        }
        JwtClaims jwtClaims = this.authTokenDecoder.decode(maybeAuthToken.get());
        UserInfo userInfo = Authenticator.toUserInfo(jwtClaims);
        String issuer = userInfo.getIssuer();
        if (!this.issuersToProviderIds.containsKey(issuer)) {
            throw new UnauthenticatedException("Unknown issuer: " + issuer);
        }
        String providerId = this.issuersToProviderIds.get(issuer);
        if (!authInfo.isProviderIdAllowed(providerId)) {
            String message = "The requested method does not allowed this provider id: " + providerId;
            throw new UnauthenticatedException(message);
        }
        this.checkJwtClaims(jwtClaims);
        Set<String> audiences = userInfo.getAudiences();
        boolean hasServiceName = audiences.contains(serviceName);
        Set<String> allowedAudiences = authInfo.getAudiencesForProvider(providerId);
        if (!hasServiceName && Sets.intersection(audiences, allowedAudiences).isEmpty()) {
            throw new UnauthenticatedException("Audiences not allowed");
        }
        return userInfo;
    }

    private void checkJwtClaims(JwtClaims jwtClaims) {
        Optional<NumericDate> expiration = Authenticator.getDateClaim("exp", jwtClaims);
        if (!expiration.isPresent()) {
            throw new UnauthenticatedException("Missing expiration field");
        }
        Optional<NumericDate> notBefore = Authenticator.getDateClaim("nbf", jwtClaims);
        NumericDate currentTime = NumericDate.fromMilliseconds(this.clock.currentTimeMillis());
        if (expiration.get().isBefore(currentTime)) {
            throw new UnauthenticatedException("The auth token has already expired");
        }
        if (notBefore.isPresent() && notBefore.get().isAfter(currentTime)) {
            String message = "Current time is earlier than the \"nbf\" time";
            throw new UnauthenticatedException(message);
        }
    }

    public static Authenticator create() {
        ServiceConfigFetcher fetcher = ServiceConfigFetcher.create();
        Service service = fetcher.fetch();
        if (!service.hasAuthentication()) {
            throw new IllegalArgumentException("Authentication is not defined in service config");
        }
        return Authenticator.create(service.getAuthentication(), Clock.SYSTEM);
    }

    @VisibleForTesting
    static Authenticator create(Authentication authentication, Clock clock) {
        List<AuthProvider> providersList = authentication.getProvidersList();
        if (providersList == null || providersList.isEmpty()) {
            throw new IllegalArgumentException("No auth providers are defined in the config.");
        }
        Map<String, IssuerKeyUrlConfig> issuerKeyConfigs = Authenticator.generateIssuerKeyConfig(providersList);
        HashMap<String, String> issuersToProviderIds = Maps.newHashMap();
        for (AuthProvider authProvider : providersList) {
            issuersToProviderIds.put(authProvider.getIssuer(), authProvider.getId());
        }
        HttpRequestFactory httpRequestFactory = new NetHttpTransport().createRequestFactory();
        DefaultKeyUriSupplier defaultKeyUriSupplier = new DefaultKeyUriSupplier(httpRequestFactory, issuerKeyConfigs);
        DefaultJwksSupplier jwksSupplier = new DefaultJwksSupplier(httpRequestFactory, defaultKeyUriSupplier);
        CachingJwksSupplier cachingJwksSupplier = new CachingJwksSupplier(jwksSupplier);
        DefaultAuthTokenVerifier authTokenVerifier = new DefaultAuthTokenVerifier(cachingJwksSupplier);
        DefaultAuthTokenDecoder authTokenDecoder = new DefaultAuthTokenDecoder(authTokenVerifier);
        CachingAuthTokenDecoder cachingAuthTokenDecoder = new CachingAuthTokenDecoder(authTokenDecoder);
        return new Authenticator(cachingAuthTokenDecoder, clock, ImmutableMap.copyOf(issuersToProviderIds));
    }

    private static Map<String, IssuerKeyUrlConfig> generateIssuerKeyConfig(List<AuthProvider> authProviders) {
        ImmutableMap.Builder<String, IssuerKeyUrlConfig> issuerConfigBuilder = ImmutableMap.builder();
        HashSet<String> issuers = Sets.newHashSet();
        for (AuthProvider authProvider : authProviders) {
            String issuer = authProvider.getIssuer();
            if (Strings.isNullOrEmpty(issuer)) {
                ((FluentLogger.Api)logger.atWarning()).log("The 'issuer' field is not set in AuthProvider (%s)", authProvider);
                continue;
            }
            if (issuers.contains(issuer)) {
                throw new IllegalArgumentException("Configuration contains multiple auth provider for the same issuer: " + issuer);
            }
            issuers.add(issuer);
            String jwksUri = authProvider.getJwksUri();
            IssuerKeyUrlConfig config = Strings.isNullOrEmpty(jwksUri) ? new IssuerKeyUrlConfig(true, Optional.absent()) : new IssuerKeyUrlConfig(false, Optional.of(new GenericUrl(jwksUri)));
            issuerConfigBuilder.put(issuer, config);
        }
        return issuerConfigBuilder.build();
    }

    private static Optional<NumericDate> getDateClaim(String claimName, JwtClaims jwtClaims) {
        try {
            NumericDate dateClaim = jwtClaims.getNumericDateClaimValue(claimName);
            return Optional.fromNullable(dateClaim);
        }
        catch (MalformedClaimException exception) {
            String message = String.format("The \"%s\" claim is malformed", claimName);
            throw new UnauthenticatedException(message);
        }
    }

    private static Optional<String> extractAuthToken(HttpServletRequest request) {
        String authHeader = request.getHeader("Authorization");
        if (authHeader != null) {
            if (authHeader.startsWith(BEARER_TOKEN_PREFIX)) {
                return Optional.of(authHeader.substring(BEARER_TOKEN_PREFIX.length()));
            }
            return Optional.absent();
        }
        String accessToken = request.getParameter(ACCESS_TOKEN_PARAM_NAME);
        if (accessToken != null) {
            return Optional.of(accessToken);
        }
        return Optional.absent();
    }

    private static UserInfo toUserInfo(JwtClaims jwtClaims) {
        try {
            List<String> audiences = jwtClaims.getAudience();
            if (audiences == null || audiences.isEmpty()) {
                throw new UnauthenticatedException("Missing audience field");
            }
            String email = jwtClaims.getClaimValue(EMAIL_CLAIM_NAME, String.class);
            String subject = jwtClaims.getSubject();
            if (subject == null) {
                throw new UnauthenticatedException("Missing subject field");
            }
            String issuer = jwtClaims.getIssuer();
            if (issuer == null) {
                throw new UnauthenticatedException("Missing issuer field");
            }
            return new UserInfo(audiences, email, subject, issuer);
        }
        catch (MalformedClaimException exception) {
            throw new UnauthenticatedException("Cannot read malformed claim", exception);
        }
    }
}

