/*
 * Decompiled with CFR 0.152.
 */
package com.google.api.control.model;

import endpoints.repackaged.com.google.api.servicecontrol.v1.Distribution;
import endpoints.repackaged.com.google.common.collect.Sets;
import endpoints.repackaged.com.google.common.flogger.FluentLogger;
import endpoints.repackaged.com.google.common.math.DoubleMath;
import endpoints.repackaged.com.google.common.primitives.Doubles;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

public final class Distributions {
    private static final String MSG_BUCKET_COUNTS_MISMATCH = "Bucket counts do not match";
    private static final String MSG_BUCKET_OPTIONS_MISMATCH = "Bucket options do not match";
    private static final String MSG_UNKNOWN_BUCKET_OPTION_TYPE = "Unknown bucket option type";
    private static final String MSG_SOME_BOUNDS_ARE_THE_SAME = "Illegal bounds, at least two bounds are the same!";
    private static final String MSG_BAD_DIST_LOW_BUCKET_COUNT = "cannot update a distribution with a low bucket count";
    private static final String MSG_DOUBLE_TOO_LOW = "%s should be > %f";
    private static final String MSG_BAD_NUM_FINITE_BUCKETS = "number of finite buckets should be > 0";
    private static final double TOLERANCE = 1.0E-5;
    private static final FluentLogger log = FluentLogger.forEnclosingClass();

    private Distributions() {
    }

    public static Distribution createExponential(int numFiniteBuckets, double growthFactor, double scale) {
        if (numFiniteBuckets <= 0) {
            throw new IllegalArgumentException(MSG_BAD_NUM_FINITE_BUCKETS);
        }
        if (growthFactor <= 1.0) {
            throw new IllegalArgumentException(String.format(MSG_DOUBLE_TOO_LOW, "growth factor", 1.0));
        }
        if (scale <= 0.0) {
            throw new IllegalArgumentException(String.format(MSG_DOUBLE_TOO_LOW, "scale", 0.0));
        }
        Distribution.ExponentialBuckets buckets = Distribution.ExponentialBuckets.newBuilder().setGrowthFactor(growthFactor).setNumFiniteBuckets(numFiniteBuckets).setScale(scale).build();
        Distribution.Builder builder = Distribution.newBuilder().setExponentialBuckets(buckets);
        for (int i = 0; i < numFiniteBuckets + 2; ++i) {
            builder.addBucketCounts(0L);
        }
        return builder.build();
    }

    public static Distribution createLinear(int numFiniteBuckets, double width, double offset) {
        if (numFiniteBuckets <= 0) {
            throw new IllegalArgumentException(MSG_BAD_NUM_FINITE_BUCKETS);
        }
        if (width <= 0.0) {
            throw new IllegalArgumentException(String.format(MSG_DOUBLE_TOO_LOW, "width", 0.0));
        }
        Distribution.LinearBuckets buckets = Distribution.LinearBuckets.newBuilder().setOffset(offset).setWidth(width).setNumFiniteBuckets(numFiniteBuckets).build();
        Distribution.Builder builder = Distribution.newBuilder().setLinearBuckets(buckets);
        for (int i = 0; i < numFiniteBuckets + 2; ++i) {
            builder.addBucketCounts(0L);
        }
        return builder.build();
    }

    public static Distribution createExplicit(double[] bounds) {
        List<Double> allBounds = Doubles.asList(bounds);
        HashSet<Double> uniqueBounds = Sets.newHashSet(allBounds);
        if (allBounds.size() != uniqueBounds.size()) {
            throw new IllegalArgumentException(MSG_SOME_BOUNDS_ARE_THE_SAME);
        }
        Collections.sort(allBounds);
        Distribution.ExplicitBuckets buckets = Distribution.ExplicitBuckets.newBuilder().addAllBounds(allBounds).build();
        Distribution.Builder builder = Distribution.newBuilder().setExplicitBuckets(buckets);
        for (int i = 0; i < allBounds.size() + 1; ++i) {
            builder.addBucketCounts(0L);
        }
        return builder.build();
    }

    public static Distribution addSample(double value, Distribution distribution) {
        Distribution.Builder builder = distribution.toBuilder();
        switch (distribution.getBucketOptionCase()) {
            case EXPLICIT_BUCKETS: {
                Distributions.updateStatistics(value, builder);
                Distributions.updateExplicitBuckets(value, builder);
                return builder.build();
            }
            case EXPONENTIAL_BUCKETS: {
                Distributions.updateStatistics(value, builder);
                Distributions.updateExponentialBuckets(value, builder);
                return builder.build();
            }
            case LINEAR_BUCKETS: {
                Distributions.updateStatistics(value, builder);
                Distributions.updateLinearBuckets(value, builder);
                return builder.build();
            }
        }
        throw new IllegalArgumentException(MSG_UNKNOWN_BUCKET_OPTION_TYPE);
    }

    public static Distribution merge(Distribution prior, Distribution latest) {
        if (!Distributions.bucketsNearlyEquals(prior, latest)) {
            throw new IllegalArgumentException(MSG_BUCKET_OPTIONS_MISMATCH);
        }
        if (prior.getBucketCountsCount() != latest.getBucketCountsCount()) {
            throw new IllegalArgumentException(MSG_BUCKET_COUNTS_MISMATCH);
        }
        if (prior.getCount() == 0L) {
            return latest;
        }
        Distribution.Builder builder = latest.toBuilder();
        long oldCount = latest.getCount();
        double oldMean = latest.getMean();
        builder.setCount(prior.getCount() + oldCount);
        builder.setMaximum(Math.max(prior.getMaximum(), latest.getMaximum()));
        builder.setMinimum(Math.min(prior.getMinimum(), latest.getMinimum()));
        double newMean = ((double)oldCount * oldMean + (double)prior.getCount() * prior.getMean()) / (double)builder.getCount();
        builder.setMean(newMean);
        double oldSumOfSquaredDeviation = latest.getSumOfSquaredDeviation();
        double newSumOfSquaredDeviation = oldSumOfSquaredDeviation + prior.getSumOfSquaredDeviation() + (double)oldCount * Math.pow(builder.getMean() - oldMean, 2.0) + (double)prior.getCount() * Math.pow(builder.getMean() - prior.getMean(), 2.0);
        builder.setSumOfSquaredDeviation(newSumOfSquaredDeviation);
        for (int i = 0; i < latest.getBucketCountsCount(); ++i) {
            builder.setBucketCounts(i, prior.getBucketCounts(i) + latest.getBucketCounts(i));
        }
        return builder.build();
    }

    private static boolean bucketsNearlyEquals(Distribution a, Distribution b) {
        Distribution.BucketOptionCase caseB;
        Distribution.BucketOptionCase caseA = a.getBucketOptionCase();
        if (caseA != (caseB = b.getBucketOptionCase())) {
            return false;
        }
        switch (caseA) {
            case EXPLICIT_BUCKETS: {
                return Distributions.bucketsNearlyEquals(a.getExplicitBuckets(), b.getExplicitBuckets());
            }
            case EXPONENTIAL_BUCKETS: {
                return Distributions.bucketsNearlyEquals(a.getExponentialBuckets(), b.getExponentialBuckets());
            }
            case LINEAR_BUCKETS: {
                return Distributions.bucketsNearlyEquals(a.getLinearBuckets(), b.getLinearBuckets());
            }
        }
        return false;
    }

    private static boolean bucketsNearlyEquals(Distribution.ExplicitBuckets a, Distribution.ExplicitBuckets b) {
        if (a.getBoundsCount() != b.getBoundsCount()) {
            return false;
        }
        for (int i = 0; i < b.getBoundsCount(); ++i) {
            if (DoubleMath.fuzzyEquals(a.getBounds(i), b.getBounds(i), 1.0E-5)) continue;
            return false;
        }
        return true;
    }

    private static boolean bucketsNearlyEquals(Distribution.ExponentialBuckets a, Distribution.ExponentialBuckets b) {
        return a.getNumFiniteBuckets() == b.getNumFiniteBuckets() && DoubleMath.fuzzyEquals(a.getGrowthFactor(), b.getGrowthFactor(), 1.0E-5) && DoubleMath.fuzzyEquals(a.getScale(), b.getScale(), 1.0E-5);
    }

    private static boolean bucketsNearlyEquals(Distribution.LinearBuckets a, Distribution.LinearBuckets b) {
        return a.getNumFiniteBuckets() == b.getNumFiniteBuckets() && DoubleMath.fuzzyEquals(a.getWidth(), b.getWidth(), 1.0E-5) && DoubleMath.fuzzyEquals(a.getOffset(), b.getOffset(), 1.0E-5);
    }

    private static void updateStatistics(double value, Distribution.Builder distribution) {
        long oldCount = distribution.getCount();
        if (oldCount == 0L) {
            distribution.setCount(1L);
            distribution.setMaximum(value);
            distribution.setMinimum(value);
            distribution.setMean(value);
            distribution.setSumOfSquaredDeviation(0.0);
        } else {
            double oldMean = distribution.getMean();
            double newMean = ((double)oldCount * oldMean + value) / (double)(oldCount + 1L);
            double deltaSumOfSquares = (value - oldMean) * (value - newMean);
            distribution.setCount(oldCount + 1L);
            distribution.setMean(newMean);
            distribution.setMaximum(Math.max(value, distribution.getMaximum()));
            distribution.setMinimum(Math.min(value, distribution.getMinimum()));
            distribution.setSumOfSquaredDeviation(deltaSumOfSquares + distribution.getSumOfSquaredDeviation());
        }
    }

    private static void updateLinearBuckets(double value, Distribution.Builder distribution) {
        Distribution.LinearBuckets buckets = distribution.getLinearBuckets();
        if (distribution.getBucketCountsCount() != buckets.getNumFiniteBuckets() + 2) {
            throw new IllegalArgumentException(MSG_BAD_DIST_LOW_BUCKET_COUNT);
        }
        double upper = buckets.getWidth() * (double)buckets.getNumFiniteBuckets() + buckets.getOffset();
        int index = 0;
        if (value >= upper) {
            index = buckets.getNumFiniteBuckets() + 1;
        } else if (value > buckets.getOffset()) {
            index = 1 + (int)Math.round((value - buckets.getOffset()) / buckets.getWidth());
        }
        long newCount = distribution.getBucketCounts(index) + 1L;
        ((FluentLogger.Api)log.atFine()).log("Updating explicit bucket %d to %d for %f", index, newCount, value);
        distribution.setBucketCounts(index, newCount);
    }

    private static void updateExponentialBuckets(double value, Distribution.Builder distribution) {
        Distribution.ExponentialBuckets buckets = distribution.getExponentialBuckets();
        if (distribution.getBucketCountsCount() != buckets.getNumFiniteBuckets() + 2) {
            throw new IllegalArgumentException(MSG_BAD_DIST_LOW_BUCKET_COUNT);
        }
        int index = 0;
        if (value > buckets.getScale()) {
            index = 1 + (int)(Math.log(value / buckets.getScale()) / Math.log(buckets.getGrowthFactor()));
            index = Math.min(buckets.getNumFiniteBuckets() + 1, index);
        }
        long newCount = distribution.getBucketCounts(index) + 1L;
        ((FluentLogger.Api)log.atFine()).log("Updating explicit bucket %d to %d for %f", index, newCount, value);
        distribution.setBucketCounts(index, newCount);
    }

    private static void updateExplicitBuckets(double value, Distribution.Builder distribution) {
        Distribution.ExplicitBuckets buckets = distribution.getExplicitBuckets();
        if (distribution.getBucketCountsCount() != buckets.getBoundsCount() + 1) {
            throw new IllegalArgumentException(MSG_BAD_DIST_LOW_BUCKET_COUNT);
        }
        int index = Collections.binarySearch(buckets.getBoundsList(), value);
        index = index < 0 ? -index - 1 : ++index;
        long newCount = distribution.getBucketCounts(index) + 1L;
        ((FluentLogger.Api)log.atFine()).log("Updating explicit bucket %d to %d for %f", index, newCount, value);
        distribution.setBucketCounts(index, newCount);
    }
}

