/*
 * Decompiled with CFR 0.152.
 */
package com.google.genai;

import com.google.auth.oauth2.GoogleCredentials;
import com.google.common.base.Ascii;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.google.genai.ApiResponse;
import com.google.genai.errors.GenAiIOException;
import com.google.genai.types.ClientOptions;
import com.google.genai.types.HttpOptions;
import java.io.IOException;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Logger;
import java.util.stream.Stream;
import okhttp3.Dispatcher;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import org.jspecify.annotations.Nullable;

abstract class ApiClient {
    private static final String SDK_VERSION = "1.8.0";
    private static final Logger logger = Logger.getLogger(ApiClient.class.getName());
    private static final ImmutableSet<String> METHODS_WITH_BODY = ImmutableSet.of((Object)"POST", (Object)"PATCH", (Object)"PUT");
    private static final ImmutableSet<String> VALID_HTTP_METHODS = ImmutableSet.builder().addAll(METHODS_WITH_BODY).add((Object)"GET").add((Object)"DELETE").build();
    final OkHttpClient httpClient;
    HttpOptions httpOptions;
    final boolean vertexAI;
    final Optional<ClientOptions> clientOptions;
    final Optional<String> apiKey;
    final Optional<String> project;
    final Optional<String> location;
    final Optional<GoogleCredentials> credentials;

    protected ApiClient(Optional<String> apiKey, Optional<HttpOptions> customHttpOptions, Optional<ClientOptions> clientOptions) {
        Preconditions.checkNotNull(apiKey, (Object)"API Key cannot be null");
        Preconditions.checkNotNull(customHttpOptions, (Object)"customHttpOptions cannot be null");
        Preconditions.checkNotNull(clientOptions, (Object)"clientOptions cannot be null");
        try {
            this.apiKey = Optional.of(apiKey.orElse(this.getApiKeyFromEnv()));
        }
        catch (NullPointerException e) {
            throw new IllegalArgumentException("API key must either be provided or set in the environment variable GOOGLE_API_KEY or GEMINI_API_KEY. If both are set, GOOGLE_API_KEY will be used.", e);
        }
        this.project = Optional.empty();
        this.location = Optional.empty();
        this.credentials = Optional.empty();
        this.vertexAI = false;
        this.clientOptions = clientOptions;
        this.httpOptions = ApiClient.defaultHttpOptions(false, this.location);
        if (customHttpOptions.isPresent()) {
            this.httpOptions = this.mergeHttpOptions(customHttpOptions.get());
        }
        this.httpClient = this.createHttpClient(this.httpOptions.timeout(), clientOptions);
    }

    ApiClient(Optional<String> project, Optional<String> location, Optional<GoogleCredentials> credentials, Optional<HttpOptions> customHttpOptions, Optional<ClientOptions> clientOptions) {
        Preconditions.checkNotNull(project, (Object)"project cannot be null");
        Preconditions.checkNotNull(location, (Object)"location cannot be null");
        Preconditions.checkNotNull(credentials, (Object)"credentials cannot be null");
        Preconditions.checkNotNull(customHttpOptions, (Object)"customHttpOptions cannot be null");
        Preconditions.checkNotNull(clientOptions, (Object)"clientOptions cannot be null");
        try {
            this.project = Optional.of(project.orElse(System.getenv("GOOGLE_CLOUD_PROJECT")));
        }
        catch (NullPointerException e) {
            throw new IllegalArgumentException("Project must either be provided or set in the environment variable GOOGLE_CLOUD_PROJECT.", e);
        }
        if (this.project.get().isEmpty()) {
            throw new IllegalArgumentException("Project must not be empty.");
        }
        try {
            this.location = Optional.of(location.orElse(System.getenv("GOOGLE_CLOUD_LOCATION")));
        }
        catch (NullPointerException e) {
            throw new IllegalArgumentException("Location must either be provided or set in the environment variable GOOGLE_CLOUD_LOCATION.", e);
        }
        if (this.location.get().isEmpty()) {
            throw new IllegalArgumentException("Location must not be empty.");
        }
        this.credentials = Optional.of(credentials.orElseGet(() -> this.defaultCredentials()));
        this.clientOptions = clientOptions;
        this.httpOptions = ApiClient.defaultHttpOptions(true, this.location);
        if (customHttpOptions.isPresent()) {
            this.httpOptions = this.mergeHttpOptions(customHttpOptions.get());
        }
        this.apiKey = Optional.empty();
        this.vertexAI = true;
        this.httpClient = this.createHttpClient(this.httpOptions.timeout(), clientOptions);
    }

    private String getApiKeyFromEnv() {
        String geminiApiKey;
        String googleApiKey = System.getenv("GOOGLE_API_KEY");
        if (googleApiKey != null && googleApiKey.isEmpty()) {
            googleApiKey = null;
        }
        if ((geminiApiKey = System.getenv("GEMINI_API_KEY")) != null && geminiApiKey.isEmpty()) {
            geminiApiKey = null;
        }
        if (googleApiKey != null && geminiApiKey != null) {
            logger.warning("Both GOOGLE_API_KEY and GEMINI_API_KEY are set. Using GOOGLE_API_KEY.");
        }
        if (googleApiKey != null) {
            return googleApiKey;
        }
        return geminiApiKey;
    }

    private OkHttpClient createHttpClient(Optional<Integer> timeout, Optional<ClientOptions> clientOptions) {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        builder.connectTimeout(Duration.ofMillis(0L));
        builder.readTimeout(Duration.ofMillis(0L));
        builder.writeTimeout(Duration.ofMillis(0L));
        timeout.ifPresent(connectTimeout -> builder.connectTimeout(Duration.ofMillis(connectTimeout.intValue())));
        clientOptions.ifPresent(options -> {
            Dispatcher dispatcher = new Dispatcher();
            options.maxConnections().ifPresent(arg_0 -> ((Dispatcher)dispatcher).setMaxRequests(arg_0));
            options.maxConnectionsPerHost().ifPresent(arg_0 -> ((Dispatcher)dispatcher).setMaxRequestsPerHost(arg_0));
            builder.dispatcher(dispatcher);
        });
        return builder.build();
    }

    protected Request buildRequest(String httpMethod, String path, String requestJson, Optional<HttpOptions> requestHttpOptions) {
        String apiVersion;
        HttpOptions mergedHttpOptions;
        String baseUrl;
        boolean queryBaseModel;
        String capitalizedHttpMethod = Ascii.toUpperCase((String)httpMethod);
        boolean bl = queryBaseModel = capitalizedHttpMethod.equals("GET") && path.startsWith("publishers/google/models");
        if (this.vertexAI() && !path.startsWith("projects/") && !queryBaseModel) {
            path = String.format("projects/%s/locations/%s/", this.project.get(), this.location.get()) + path;
        }
        if ((baseUrl = (mergedHttpOptions = this.mergeHttpOptions(requestHttpOptions.orElse(null))).baseUrl().orElseThrow(() -> new IllegalStateException("baseUrl is required."))).endsWith("/")) {
            baseUrl = baseUrl.substring(0, baseUrl.length() - 1);
        }
        String requestUrl = (apiVersion = mergedHttpOptions.apiVersion().orElseThrow(() -> new IllegalStateException("apiVersion is required."))).isEmpty() ? String.format("%s/%s", baseUrl, path) : String.format("%s/%s/%s", baseUrl, apiVersion, path);
        if (!VALID_HTTP_METHODS.contains((Object)capitalizedHttpMethod)) {
            throw new IllegalArgumentException("Unsupported HTTP method: " + capitalizedHttpMethod);
        }
        RequestBody body = METHODS_WITH_BODY.contains((Object)capitalizedHttpMethod) ? RequestBody.create((String)requestJson, (MediaType)MediaType.parse((String)"application/json")) : null;
        Request.Builder requestBuilder = new Request.Builder().url(requestUrl).method(capitalizedHttpMethod, body);
        this.setHeaders(requestBuilder, mergedHttpOptions);
        return requestBuilder.build();
    }

    protected Request buildRequest(String httpMethod, String url, byte[] requestBytes, Optional<HttpOptions> requestHttpOptions) {
        HttpOptions mergedHttpOptions = this.mergeHttpOptions(requestHttpOptions.orElse(null));
        if (httpMethod.equalsIgnoreCase("POST")) {
            RequestBody body = RequestBody.create((MediaType)MediaType.get((String)"application/octet-stream"), (byte[])requestBytes);
            Request.Builder requestBuilder = new Request.Builder().url(url).post(body);
            this.setHeaders(requestBuilder, mergedHttpOptions);
            return requestBuilder.build();
        }
        throw new IllegalArgumentException("The request method with bytes is only supported for POST. Unsupported HTTP method: " + httpMethod);
    }

    private void setHeaders(Request.Builder request, HttpOptions requestHttpOptions) {
        for (Map.Entry<String, String> header : requestHttpOptions.headers().orElse((Map<String, String>)ImmutableMap.of()).entrySet()) {
            request.header(header.getKey(), header.getValue());
        }
        if (this.apiKey.isPresent()) {
            request.header("x-goog-api-key", this.apiKey.get());
        } else {
            String accessToken;
            GoogleCredentials cred = this.credentials.orElseThrow(() -> new IllegalStateException("credentials is required"));
            try {
                cred.refreshIfExpired();
            }
            catch (IOException e) {
                throw new GenAiIOException("Failed to refresh credentials.", e);
            }
            try {
                accessToken = cred.getAccessToken().getTokenValue();
            }
            catch (NullPointerException e) {
                if (e.getMessage().contains("because the return value of \"com.google.auth.oauth2.GoogleCredentials.getAccessToken()\" is null")) {
                    accessToken = "";
                }
                throw e;
            }
            request.header("Authorization", "Bearer " + accessToken);
            if (cred.getQuotaProjectId() != null) {
                request.header("x-goog-user-project", cred.getQuotaProjectId());
            }
        }
    }

    @CanIgnoreReturnValue
    public abstract ApiResponse request(String var1, String var2, String var3, Optional<HttpOptions> var4);

    public abstract ApiResponse request(String var1, String var2, byte[] var3, Optional<HttpOptions> var4);

    static String libraryVersion() {
        String libraryLabel = String.format("google-genai-sdk/%s", SDK_VERSION);
        String languageLabel = "gl-java/" + System.getProperty("java.version");
        return libraryLabel + " " + languageLabel;
    }

    public boolean vertexAI() {
        return this.vertexAI;
    }

    public @Nullable String project() {
        return this.project.orElse(null);
    }

    public @Nullable String location() {
        return this.location.orElse(null);
    }

    public @Nullable String apiKey() {
        return this.apiKey.orElse(null);
    }

    OkHttpClient httpClient() {
        return this.httpClient;
    }

    private Optional<Map<String, String>> getTimeoutHeader(HttpOptions httpOptionsToApply) {
        if (httpOptionsToApply.timeout().isPresent()) {
            int timeoutInSeconds = (int)Math.ceil((double)httpOptionsToApply.timeout().get().intValue() / 1000.0);
            return Optional.of(ImmutableMap.of((Object)"X-Server-Timeout", (Object)Integer.toString(timeoutInSeconds)));
        }
        return Optional.empty();
    }

    HttpOptions mergeHttpOptions(HttpOptions httpOptionsToApply) {
        if (httpOptionsToApply == null) {
            return this.httpOptions;
        }
        HttpOptions.Builder mergedHttpOptionsBuilder = this.httpOptions.toBuilder();
        if (httpOptionsToApply.baseUrl().isPresent()) {
            mergedHttpOptionsBuilder.baseUrl(httpOptionsToApply.baseUrl().get());
        }
        if (httpOptionsToApply.apiVersion().isPresent()) {
            mergedHttpOptionsBuilder.apiVersion(httpOptionsToApply.apiVersion().get());
        }
        if (httpOptionsToApply.timeout().isPresent()) {
            mergedHttpOptionsBuilder.timeout(httpOptionsToApply.timeout().get());
        }
        if (httpOptionsToApply.headers().isPresent()) {
            Stream headersStream = Stream.concat(Stream.concat(this.httpOptions.headers().orElse((Map<String, String>)ImmutableMap.of()).entrySet().stream(), this.getTimeoutHeader(httpOptionsToApply).orElse((Map<String, String>)ImmutableMap.of()).entrySet().stream()), httpOptionsToApply.headers().orElse((Map<String, String>)ImmutableMap.of()).entrySet().stream());
            Map mergedHeaders = (Map)headersStream.collect(ImmutableMap.toImmutableMap(Map.Entry::getKey, Map.Entry::getValue, (val1, val2) -> val2));
            mergedHttpOptionsBuilder.headers(mergedHeaders);
        }
        return mergedHttpOptionsBuilder.build();
    }

    static HttpOptions defaultHttpOptions(boolean vertexAI, Optional<String> location) {
        ImmutableMap.Builder defaultHeaders = ImmutableMap.builder();
        defaultHeaders.put((Object)"Content-Type", (Object)"application/json");
        defaultHeaders.put((Object)"user-agent", (Object)ApiClient.libraryVersion());
        defaultHeaders.put((Object)"x-goog-api-client", (Object)ApiClient.libraryVersion());
        HttpOptions.Builder defaultHttpOptionsBuilder = HttpOptions.builder().headers((Map<String, String>)defaultHeaders.build());
        if (vertexAI && location.isPresent()) {
            defaultHttpOptionsBuilder.baseUrl(location.get().equalsIgnoreCase("global") ? "https://aiplatform.googleapis.com" : String.format("https://%s-aiplatform.googleapis.com", location.get())).apiVersion("v1beta1");
        } else {
            if (vertexAI && !location.isPresent()) {
                throw new IllegalArgumentException("Location must be provided for Vertex AI APIs.");
            }
            defaultHttpOptionsBuilder.baseUrl("https://generativelanguage.googleapis.com").apiVersion("v1beta");
        }
        return defaultHttpOptionsBuilder.build();
    }

    GoogleCredentials defaultCredentials() {
        try {
            return GoogleCredentials.getApplicationDefault().createScoped(new String[]{"https://www.googleapis.com/auth/cloud-platform"});
        }
        catch (IOException e) {
            throw new GenAiIOException("Failed to get application default credentials, please explicitly provide credentials.", e);
        }
    }
}

