/*
 * Copyright (C) 2012 The Guava Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.common.collect.testing.google;

import static com.google.common.collect.testing.features.CollectionSize.ZERO;
import static com.google.common.collect.testing.features.MapFeature.SUPPORTS_REMOVE;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.collect.testing.Helpers;
import com.google.common.collect.testing.features.CollectionSize;
import com.google.common.collect.testing.features.MapFeature;

import java.lang.reflect.Method;
import java.util.Iterator;

/**
 * Tester for {@code BiMap.remove}.
 *
 * @author Louis Wasserman
 */
@GwtCompatible(emulated = true)
public class BiMapRemoveTester<K, V> extends AbstractBiMapTester<K, V> {
  @SuppressWarnings("unchecked")
  @MapFeature.Require(SUPPORTS_REMOVE)
  @CollectionSize.Require(absent = ZERO)
  public void testRemoveKeyRemovesFromInverse() {
    getMap().remove(samples.e0.getKey());
    expectMissing(samples.e0);
  }

  @SuppressWarnings("unchecked")
  @MapFeature.Require(SUPPORTS_REMOVE)
  @CollectionSize.Require(absent = ZERO)
  public void testRemoveKeyFromKeySetRemovesFromInverse() {
    getMap().keySet().remove(samples.e0.getKey());
    expectMissing(samples.e0);
  }

  @SuppressWarnings("unchecked")
  @MapFeature.Require(SUPPORTS_REMOVE)
  @CollectionSize.Require(absent = ZERO)
  public void testRemoveFromValuesRemovesFromInverse() {
    getMap().values().remove(samples.e0.getValue());
    expectMissing(samples.e0);
  }

  @SuppressWarnings("unchecked")
  @MapFeature.Require(SUPPORTS_REMOVE)
  @CollectionSize.Require(absent = ZERO)
  public void testRemoveFromInverseRemovesFromForward() {
    getMap().inverse().remove(samples.e0.getValue());
    expectMissing(samples.e0);
  }

  @SuppressWarnings("unchecked")
  @MapFeature.Require(SUPPORTS_REMOVE)
  @CollectionSize.Require(absent = ZERO)
  public void testRemoveFromInverseKeySetRemovesFromForward() {
    getMap().inverse().keySet().remove(samples.e0.getValue());
    expectMissing(samples.e0);
  }

  @SuppressWarnings("unchecked")
  @MapFeature.Require(SUPPORTS_REMOVE)
  @CollectionSize.Require(absent = ZERO)
  public void testRemoveFromInverseValuesRemovesFromInverse() {
    getMap().inverse().values().remove(samples.e0.getKey());
    expectMissing(samples.e0);
  }

  @MapFeature.Require(SUPPORTS_REMOVE)
  @CollectionSize.Require(absent = ZERO)
  public void testKeySetIteratorRemove() {
    int initialSize = getNumElements();
    Iterator<K> iterator = getMap().keySet().iterator();
    iterator.next();
    iterator.remove();
    assertEquals(initialSize - 1, getMap().size());
    assertEquals(initialSize - 1, getMap().inverse().size());
  }

  /**
   * Returns the {@link Method} instance for
   * {@link #testKeySetIteratorRemove()} so that tests of
   * {@code Maps.filterEntries(BiMap, Predicate)} can suppress
   * it with {@code FeatureSpecificTestSuiteBuilder.suppressing()}.
   */
  @GwtIncompatible("reflection")
  public static Method getKeySetIteratorRemoveMethod() {
    return Helpers.getMethod(BiMapRemoveTester.class, "testKeySetIteratorRemove");
  }
}
