/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.hash.Funnel;
import com.google.common.hash.HashCode;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.PrimitiveSink;
import com.google.common.jdk5backport.Arrays;
import com.google.common.primitives.Ints;
import com.google.common.testing.EqualsTester;
import java.nio.charset.Charset;
import java.util.HashSet;
import java.util.Random;
import org.junit.Assert;

final class HashTestUtils {
    static final Funnel<Object> BAD_FUNNEL = new Funnel<Object>(){

        public void funnel(Object object, PrimitiveSink bytePrimitiveSink) {
            bytePrimitiveSink.putInt(object.hashCode());
        }
    };
    private static final ImmutableSet<Charset> CHARSETS = ImmutableSet.of((Object)Charsets.ISO_8859_1, (Object)Charsets.US_ASCII, (Object)Charsets.UTF_16, (Object)Charsets.UTF_16BE, (Object)Charsets.UTF_16LE, (Object)Charsets.UTF_8, (Object[])new Charset[0]);

    private HashTestUtils() {
    }

    static byte[] ascii(String string) {
        byte[] bytes = new byte[string.length()];
        for (int i = 0; i < string.length(); ++i) {
            bytes[i] = (byte)string.charAt(i);
        }
        return bytes;
    }

    static void verifyHashFunction(HashFn hashFunction, int hashbits, int expected) {
        int hashBytes = hashbits / 8;
        byte[] key = new byte[256];
        byte[] hashes = new byte[hashBytes * 256];
        for (int i = 0; i < 256; ++i) {
            key[i] = (byte)i;
            int seed = 256 - i;
            byte[] hash = hashFunction.hash(Arrays.copyOf(key, i), seed);
            System.arraycopy(hash, 0, hashes, i * hashBytes, hash.length);
        }
        byte[] result = hashFunction.hash(hashes, 0);
        int verification = Integer.reverseBytes(Ints.fromByteArray((byte[])result));
        if (expected != verification) {
            throw new AssertionError((Object)("Expected: " + Integer.toHexString(expected) + " got: " + Integer.toHexString(verification)));
        }
    }

    static void checkNoFunnels(HashFunction function) {
        Random rand = new Random(0L);
        int keyBits = 32;
        int hashBits = function.bits();
        for (int i = 0; i < keyBits; ++i) {
            int hash2;
            int hash1;
            int same = 0;
            int count = 0;
            int maxCount = (int)(4.0 * Math.log(2 * keyBits * hashBits) + 1.0);
            for (int diff = 0; same != -1 || diff != -1; same |= ~(hash1 ^ hash2), diff |= hash1 ^ hash2) {
                int key1 = rand.nextInt();
                int key2 = key1 ^ 1 << i;
                hash1 = function.hashInt(key1).asInt();
                hash2 = function.hashInt(key2).asInt();
                if (++count <= maxCount) continue;
                Assert.fail((String)("input bit(" + i + ") was found not to affect all " + hashBits + " output bits; The unaffected bits are " + "as follows: " + ~(same & diff) + ". This was " + "determined after " + count + " trials."));
            }
        }
    }

    static void checkAvalanche(HashFunction function, int trials, double epsilon) {
        Random rand = new Random(0L);
        int keyBits = 32;
        int hashBits = function.bits();
        for (int i = 0; i < keyBits; ++i) {
            int j;
            int[] same = new int[hashBits];
            int[] diff = new int[hashBits];
            for (j = 0; j < trials; ++j) {
                int key1 = rand.nextInt();
                int key2 = key1 ^ 1 << i;
                int hash1 = function.hashInt(key1).asInt();
                int hash2 = function.hashInt(key2).asInt();
                for (int k = 0; k < hashBits; ++k) {
                    if ((hash1 & 1 << k) == (hash2 & 1 << k)) {
                        int n = k;
                        same[n] = same[n] + 1;
                        continue;
                    }
                    int n = k;
                    diff[n] = diff[n] + 1;
                }
            }
            for (j = 0; j < hashBits; ++j) {
                double prob = (double)diff[j] / (double)(diff[j] + same[j]);
                Assert.assertEquals((double)0.5, (double)prob, (double)epsilon);
            }
        }
    }

    static void checkNo2BitCharacteristics(HashFunction function) {
        Random rand = new Random(0L);
        int keyBits = 32;
        for (int i = 0; i < keyBits; ++i) {
            for (int j = 0; j < keyBits; ++j) {
                if (j <= i) continue;
                int count = 0;
                int maxCount = 20;
                boolean diff = false;
                while (!diff) {
                    int hash2;
                    int delta = 1 << i | 1 << j;
                    int key1 = rand.nextInt();
                    int key2 = key1 ^ delta;
                    int hash1 = function.hashInt(key1).asInt();
                    if ((hash1 ^ (hash2 = function.hashInt(key2).asInt())) != delta) {
                        diff = true;
                        continue;
                    }
                    if (++count <= maxCount) continue;
                    Assert.fail((String)("2-bit delta (" + i + ", " + j + ") is likely a " + "characteristic for this hash. This was " + "determined after " + count + " trials"));
                }
            }
        }
    }

    static void check2BitAvalanche(HashFunction function, int trials, double epsilon) {
        Random rand = new Random(0L);
        int keyBits = 32;
        int hashBits = function.bits();
        for (int bit1 = 0; bit1 < keyBits; ++bit1) {
            for (int bit2 = 0; bit2 < keyBits; ++bit2) {
                int j;
                if (bit2 <= bit1) continue;
                int delta = 1 << bit1 | 1 << bit2;
                int[] same = new int[hashBits];
                int[] diff = new int[hashBits];
                for (j = 0; j < trials; ++j) {
                    int key1 = rand.nextInt();
                    int key2 = key1 ^ delta;
                    int hash1 = function.hashInt(key1).asInt();
                    int hash2 = function.hashInt(key2).asInt();
                    for (int k = 0; k < hashBits; ++k) {
                        if ((hash1 & 1 << k) == (hash2 & 1 << k)) {
                            int n = k;
                            same[n] = same[n] + 1;
                            continue;
                        }
                        int n = k;
                        diff[n] = diff[n] + 1;
                    }
                }
                for (j = 0; j < hashBits; ++j) {
                    double prob = (double)diff[j] / (double)(diff[j] + same[j]);
                    Assert.assertEquals((double)0.5, (double)prob, (double)epsilon);
                }
            }
        }
    }

    static void assertInvariants(HashFunction hashFunction) {
        int objects = 100;
        HashSet hashcodes = Sets.newHashSetWithExpectedSize((int)objects);
        for (int i = 0; i < objects; ++i) {
            Object o = new Object();
            HashCode hashcode1 = hashFunction.hashObject(o, BAD_FUNNEL);
            HashCode hashcode2 = hashFunction.hashObject(o, BAD_FUNNEL);
            Assert.assertEquals((Object)hashcode1, (Object)hashcode2);
            Assert.assertEquals((long)hashFunction.bits(), (long)hashcode1.bits());
            Assert.assertEquals((long)hashFunction.bits(), (long)(hashcode1.asBytes().length * 8));
            hashcodes.add(hashcode1);
        }
        Assert.assertTrue(((double)hashcodes.size() > (double)objects * 0.95 ? 1 : 0) != 0);
        HashTestUtils.assertHashBytesThrowsCorrectExceptions(hashFunction);
        HashTestUtils.assertIndependentHashers(hashFunction);
        HashTestUtils.assertShortcutsAreEquivalent(hashFunction, 512);
    }

    static void assertHashBytesThrowsCorrectExceptions(HashFunction hashFunction) {
        hashFunction.hashBytes(new byte[64], 0, 0);
        try {
            hashFunction.hashBytes(new byte[128], -1, 128);
            Assert.fail();
        }
        catch (IndexOutOfBoundsException expected) {
            // empty catch block
        }
        try {
            hashFunction.hashBytes(new byte[128], 64, 256);
            Assert.fail();
        }
        catch (IndexOutOfBoundsException expected) {
            // empty catch block
        }
        try {
            hashFunction.hashBytes(new byte[64], 0, -1);
            Assert.fail();
        }
        catch (IndexOutOfBoundsException indexOutOfBoundsException) {
            // empty catch block
        }
    }

    static void assertIndependentHashers(HashFunction hashFunction) {
        int numActions = 100;
        HashCode expected1 = HashTestUtils.randomHash(hashFunction, new Random(1L), numActions);
        HashCode expected2 = HashTestUtils.randomHash(hashFunction, new Random(2L), numActions);
        Random random1 = new Random(1L);
        Random random2 = new Random(2L);
        Hasher hasher1 = hashFunction.newHasher();
        Hasher hasher2 = hashFunction.newHasher();
        for (int i = 0; i < numActions; ++i) {
            RandomHasherAction.pickAtRandom(random1).performAction(random1, (Iterable<? extends PrimitiveSink>)ImmutableSet.of((Object)hasher1));
            RandomHasherAction.pickAtRandom(random2).performAction(random2, (Iterable<? extends PrimitiveSink>)ImmutableSet.of((Object)hasher2));
        }
        Assert.assertEquals((Object)expected1, (Object)hasher1.hash());
        Assert.assertEquals((Object)expected2, (Object)hasher2.hash());
    }

    static HashCode randomHash(HashFunction hashFunction, Random random, int numActions) {
        Hasher hasher = hashFunction.newHasher();
        for (int i = 0; i < numActions; ++i) {
            RandomHasherAction.pickAtRandom(random).performAction(random, (Iterable<? extends PrimitiveSink>)ImmutableSet.of((Object)hasher));
        }
        return hasher.hash();
    }

    private static void assertShortcutsAreEquivalent(HashFunction hashFunction, int trials) {
        Random random = new Random(42085L);
        for (int i = 0; i < trials; ++i) {
            HashTestUtils.assertHashBytesEquivalence(hashFunction, random);
            HashTestUtils.assertHashIntEquivalence(hashFunction, random);
            HashTestUtils.assertHashLongEquivalence(hashFunction, random);
            HashTestUtils.assertHashStringEquivalence(hashFunction, random);
            HashTestUtils.assertHashStringWithSurrogatesEquivalence(hashFunction, random);
        }
    }

    private static void assertHashBytesEquivalence(HashFunction hashFunction, Random random) {
        int size = random.nextInt(2048);
        byte[] bytes = new byte[size];
        random.nextBytes(bytes);
        Assert.assertEquals((Object)hashFunction.hashBytes(bytes), (Object)hashFunction.newHasher(size).putBytes(bytes).hash());
        int off = random.nextInt(size);
        int len = random.nextInt(size - off);
        Assert.assertEquals((Object)hashFunction.hashBytes(bytes, off, len), (Object)hashFunction.newHasher(size).putBytes(bytes, off, len).hash());
    }

    private static void assertHashIntEquivalence(HashFunction hashFunction, Random random) {
        int i = random.nextInt();
        Assert.assertEquals((Object)hashFunction.hashInt(i), (Object)hashFunction.newHasher().putInt(i).hash());
    }

    private static void assertHashLongEquivalence(HashFunction hashFunction, Random random) {
        long l = random.nextLong();
        Assert.assertEquals((Object)hashFunction.hashLong(l), (Object)hashFunction.newHasher().putLong(l).hash());
    }

    private static void assertHashStringEquivalence(HashFunction hashFunction, Random random) {
        new EqualsTester().addEqualityGroup(new Object[]{hashFunction.hashUnencodedChars((CharSequence)"abc"), hashFunction.newHasher().putUnencodedChars((CharSequence)"abc").hash(), hashFunction.newHasher().putUnencodedChars((CharSequence)"ab").putUnencodedChars((CharSequence)"c").hash(), hashFunction.newHasher().putUnencodedChars((CharSequence)"a").putUnencodedChars((CharSequence)"bc").hash(), hashFunction.newHasher().putUnencodedChars((CharSequence)"a").putUnencodedChars((CharSequence)"b").putUnencodedChars((CharSequence)"c").hash(), hashFunction.newHasher().putChar('a').putUnencodedChars((CharSequence)"bc").hash(), hashFunction.newHasher().putUnencodedChars((CharSequence)"ab").putChar('c').hash(), hashFunction.newHasher().putChar('a').putChar('b').putChar('c').hash()}).testEquals();
        int size = random.nextInt(2048);
        byte[] bytes = new byte[size];
        random.nextBytes(bytes);
        String string = new String(bytes);
        Assert.assertEquals((Object)hashFunction.hashUnencodedChars((CharSequence)string), (Object)hashFunction.newHasher().putUnencodedChars((CharSequence)string).hash());
        for (Charset charset : CHARSETS) {
            Assert.assertEquals((Object)hashFunction.hashString((CharSequence)string, charset), (Object)hashFunction.newHasher().putString((CharSequence)string, charset).hash());
        }
    }

    private static void assertHashStringWithSurrogatesEquivalence(HashFunction hashFunction, Random random) {
        int size = random.nextInt(8) + 1;
        char[] chars = new char[size];
        for (int i = 0; i < chars.length; ++i) {
            chars[i] = random.nextBoolean() ? HashTestUtils.randomLowSurrogate(random) : HashTestUtils.randomHighSurrogate(random);
        }
        String string = new String(chars);
        Assert.assertEquals((Object)hashFunction.hashUnencodedChars((CharSequence)string), (Object)hashFunction.newHasher().putUnencodedChars((CharSequence)string).hash());
    }

    static char randomLowSurrogate(Random random) {
        return (char)(56320 + random.nextInt(1024));
    }

    static char randomHighSurrogate(Random random) {
        return (char)(55296 + random.nextInt(1024));
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    static enum RandomHasherAction {
        PUT_BOOLEAN{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                boolean value = random.nextBoolean();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putBoolean(value);
                }
            }
        }
        ,
        PUT_BYTE{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                int value = random.nextInt();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putByte((byte)value);
                }
            }
        }
        ,
        PUT_SHORT{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                short value = (short)random.nextInt();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putShort(value);
                }
            }
        }
        ,
        PUT_CHAR{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                char value = (char)random.nextInt();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putChar(value);
                }
            }
        }
        ,
        PUT_INT{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                int value = random.nextInt();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putInt(value);
                }
            }
        }
        ,
        PUT_LONG{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                long value = random.nextLong();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putLong(value);
                }
            }
        }
        ,
        PUT_FLOAT{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                float value = random.nextFloat();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putFloat(value);
                }
            }
        }
        ,
        PUT_DOUBLE{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                double value = random.nextDouble();
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putDouble(value);
                }
            }
        }
        ,
        PUT_BYTES{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                byte[] value = new byte[random.nextInt(128)];
                random.nextBytes(value);
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putBytes(value);
                }
            }
        }
        ,
        PUT_BYTES_INT_INT{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                byte[] value = new byte[random.nextInt(128)];
                random.nextBytes(value);
                int off = random.nextInt(value.length + 1);
                int len = random.nextInt(value.length - off + 1);
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putBytes(value);
                }
            }
        }
        ,
        PUT_STRING{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                char[] value = new char[random.nextInt(128)];
                for (int i = 0; i < value.length; ++i) {
                    value[i] = (char)random.nextInt();
                }
                String s = new String(value);
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putUnencodedChars((CharSequence)s);
                }
            }
        }
        ,
        PUT_STRING_LOW_SURROGATE{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                String s = new String(new char[]{HashTestUtils.randomLowSurrogate(random)});
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putUnencodedChars((CharSequence)s);
                }
            }
        }
        ,
        PUT_STRING_HIGH_SURROGATE{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                String s = new String(new char[]{HashTestUtils.randomHighSurrogate(random)});
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putUnencodedChars((CharSequence)s);
                }
            }
        }
        ,
        PUT_STRING_LOW_HIGH_SURROGATE{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                String s = new String(new char[]{HashTestUtils.randomLowSurrogate(random), HashTestUtils.randomHighSurrogate(random)});
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putUnencodedChars((CharSequence)s);
                }
            }
        }
        ,
        PUT_STRING_HIGH_LOW_SURROGATE{

            @Override
            void performAction(Random random, Iterable<? extends PrimitiveSink> sinks) {
                String s = new String(new char[]{HashTestUtils.randomHighSurrogate(random), HashTestUtils.randomLowSurrogate(random)});
                for (PrimitiveSink primitiveSink : sinks) {
                    primitiveSink.putUnencodedChars((CharSequence)s);
                }
            }
        };

        private static final RandomHasherAction[] actions;

        abstract void performAction(Random var1, Iterable<? extends PrimitiveSink> var2);

        static RandomHasherAction pickAtRandom(Random random) {
            return actions[random.nextInt(actions.length)];
        }

        static {
            actions = RandomHasherAction.values();
        }
    }

    static interface HashFn {
        public byte[] hash(byte[] var1, int var2);
    }
}

