/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.io;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.hash.Hashing;
import com.google.common.io.ByteProcessor;
import com.google.common.io.ByteSink;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteSourceTester;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.google.common.io.Closer;
import com.google.common.io.CloserTest;
import com.google.common.io.IoTestCase;
import com.google.common.io.SourceSinkFactories;
import com.google.common.io.TestByteSink;
import com.google.common.io.TestByteSource;
import com.google.common.io.TestOption;
import com.google.common.io.TestOutputStream;
import com.google.common.testing.TestLogHandler;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.logging.Handler;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.junit.Assert;

public class ByteSourceTest
extends IoTestCase {
    private static final byte[] bytes = ByteSourceTest.newPreFilledByteArray(10000);
    private TestByteSource source;
    private static final ByteSource BROKEN_CLOSE_SOURCE = new TestByteSource(new byte[10], TestOption.CLOSE_THROWS);
    private static final ByteSource BROKEN_OPEN_SOURCE = new TestByteSource(new byte[10], TestOption.OPEN_THROWS);
    private static final ByteSource BROKEN_READ_SOURCE = new TestByteSource(new byte[10], TestOption.READ_THROWS);
    private static final ByteSink BROKEN_CLOSE_SINK = new TestByteSink(TestOption.CLOSE_THROWS);
    private static final ByteSink BROKEN_OPEN_SINK = new TestByteSink(TestOption.OPEN_THROWS);
    private static final ByteSink BROKEN_WRITE_SINK = new TestByteSink(TestOption.WRITE_THROWS);
    private static final ImmutableSet<ByteSource> BROKEN_SOURCES = ImmutableSet.of((Object)BROKEN_CLOSE_SOURCE, (Object)BROKEN_OPEN_SOURCE, (Object)BROKEN_READ_SOURCE);
    private static final ImmutableSet<ByteSink> BROKEN_SINKS = ImmutableSet.of((Object)BROKEN_CLOSE_SINK, (Object)BROKEN_OPEN_SINK, (Object)BROKEN_WRITE_SINK);

    public static TestSuite suite() {
        TestSuite suite = new TestSuite();
        suite.addTest((Test)ByteSourceTester.tests("ByteSource.wrap[byte[]]", SourceSinkFactories.byteArraySourceFactory(), true));
        suite.addTest((Test)ByteSourceTester.tests("ByteSource.empty[]", SourceSinkFactories.emptyByteSourceFactory(), true));
        suite.addTestSuite(ByteSourceTest.class);
        return suite;
    }

    protected void setUp() throws Exception {
        this.source = new TestByteSource(bytes, new TestOption[0]);
    }

    public void testOpenBufferedStream() throws IOException {
        InputStream in = this.source.openBufferedStream();
        ByteSourceTest.assertTrue((boolean)this.source.wasStreamOpened());
        ByteSourceTest.assertFalse((boolean)this.source.wasStreamClosed());
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ByteStreams.copy((InputStream)in, (OutputStream)out);
        in.close();
        out.close();
        ByteSourceTest.assertTrue((boolean)this.source.wasStreamClosed());
        Assert.assertArrayEquals((byte[])bytes, (byte[])out.toByteArray());
    }

    public void testSize() throws IOException {
        ByteSourceTest.assertEquals((long)bytes.length, (long)this.source.size());
        ByteSourceTest.assertTrue((this.source.wasStreamOpened() && this.source.wasStreamClosed() ? 1 : 0) != 0);
        ByteSourceTest.assertEquals((long)bytes.length, (long)new TestByteSource(bytes, TestOption.SKIP_THROWS).size());
        ByteSourceTest.assertEquals((long)bytes.length, (long)new TestByteSource(bytes, TestOption.AVAILABLE_ALWAYS_ZERO).size());
    }

    public void testCopyTo_outputStream() throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ByteSourceTest.assertEquals((long)bytes.length, (long)this.source.copyTo(out));
        ByteSourceTest.assertTrue((this.source.wasStreamOpened() && this.source.wasStreamClosed() ? 1 : 0) != 0);
        Assert.assertArrayEquals((byte[])bytes, (byte[])out.toByteArray());
    }

    public void testCopyTo_byteSink() throws IOException {
        TestByteSink sink = new TestByteSink(new TestOption[0]);
        ByteSourceTest.assertFalse((sink.wasStreamOpened() || sink.wasStreamClosed() ? 1 : 0) != 0);
        ByteSourceTest.assertEquals((long)bytes.length, (long)this.source.copyTo(sink));
        ByteSourceTest.assertTrue((this.source.wasStreamOpened() && this.source.wasStreamClosed() ? 1 : 0) != 0);
        ByteSourceTest.assertTrue((sink.wasStreamOpened() && sink.wasStreamClosed() ? 1 : 0) != 0);
        Assert.assertArrayEquals((byte[])bytes, (byte[])sink.getBytes());
    }

    public void testRead_toArray() throws IOException {
        Assert.assertArrayEquals((byte[])bytes, (byte[])this.source.read());
        ByteSourceTest.assertTrue((this.source.wasStreamOpened() && this.source.wasStreamClosed() ? 1 : 0) != 0);
    }

    public void testRead_withProcessor() throws IOException {
        final byte[] processedBytes = new byte[bytes.length];
        ByteProcessor<byte[]> processor = new ByteProcessor<byte[]>(){
            int pos;

            public boolean processBytes(byte[] buf, int off, int len) throws IOException {
                System.arraycopy(buf, off, processedBytes, this.pos, len);
                this.pos += len;
                return true;
            }

            public byte[] getResult() {
                return processedBytes;
            }
        };
        this.source.read((ByteProcessor)processor);
        ByteSourceTest.assertTrue((this.source.wasStreamOpened() && this.source.wasStreamClosed() ? 1 : 0) != 0);
        Assert.assertArrayEquals((byte[])bytes, (byte[])processedBytes);
    }

    public void testRead_withProcessor_stopsOnFalse() throws IOException {
        ByteProcessor<Void> processor = new ByteProcessor<Void>(){
            boolean firstCall = true;

            public boolean processBytes(byte[] buf, int off, int len) throws IOException {
                junit.framework.Assert.assertTrue((String)"consume() called twice", (boolean)this.firstCall);
                this.firstCall = false;
                return false;
            }

            public Void getResult() {
                return null;
            }
        };
        this.source.read((ByteProcessor)processor);
        ByteSourceTest.assertTrue((this.source.wasStreamOpened() && this.source.wasStreamClosed() ? 1 : 0) != 0);
    }

    public void testHash() throws IOException {
        TestByteSource byteSource = new TestByteSource("hamburger\n".getBytes(Charsets.US_ASCII.name()), new TestOption[0]);
        ByteSourceTest.assertEquals((String)"cfa0c5002275c90508338a5cdb2a9781", (String)byteSource.hash(Hashing.md5()).toString());
    }

    public void testContentEquals() throws IOException {
        ByteSourceTest.assertTrue((boolean)this.source.contentEquals(this.source));
        ByteSourceTest.assertTrue((this.source.wasStreamOpened() && this.source.wasStreamClosed() ? 1 : 0) != 0);
        TestByteSource equalSource = new TestByteSource(bytes, new TestOption[0]);
        ByteSourceTest.assertTrue((boolean)this.source.contentEquals(equalSource));
        ByteSourceTest.assertTrue((boolean)new TestByteSource(bytes, new TestOption[0]).contentEquals(this.source));
        TestByteSource fewerBytes = new TestByteSource(ByteSourceTest.newPreFilledByteArray(bytes.length / 2), new TestOption[0]);
        ByteSourceTest.assertFalse((boolean)this.source.contentEquals(fewerBytes));
        byte[] copy = (byte[])bytes.clone();
        copy[9876] = 1;
        TestByteSource oneByteOff = new TestByteSource(copy, new TestOption[0]);
        ByteSourceTest.assertFalse((boolean)this.source.contentEquals(oneByteOff));
    }

    public void testSlice() throws IOException {
        try {
            this.source.slice(-1L, 10L);
            ByteSourceTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            this.source.slice(0L, -1L);
            ByteSourceTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        ByteSourceTest.assertCorrectSlice(0, 0, 0L, 0);
        ByteSourceTest.assertCorrectSlice(0, 0, 1L, 0);
        ByteSourceTest.assertCorrectSlice(100, 0, 10L, 10);
        ByteSourceTest.assertCorrectSlice(100, 0, 100L, 100);
        ByteSourceTest.assertCorrectSlice(100, 5, 10L, 10);
        ByteSourceTest.assertCorrectSlice(100, 5, 100L, 95);
        ByteSourceTest.assertCorrectSlice(100, 100, 0L, 0);
        ByteSourceTest.assertCorrectSlice(100, 100, 10L, 0);
        try {
            ByteSourceTest.assertCorrectSlice(100, 101, 10L, 0);
            ByteSourceTest.fail();
        }
        catch (EOFException eOFException) {
            // empty catch block
        }
    }

    private static void assertCorrectSlice(int input, int offset, long length, int expectRead) throws IOException {
        Preconditions.checkArgument((expectRead == (int)Math.max(0L, Math.min((long)input, (long)offset + length) - (long)offset) ? 1 : 0) != 0);
        byte[] expected = ByteSourceTest.newPreFilledByteArray(offset, expectRead);
        TestByteSource source = new TestByteSource(ByteSourceTest.newPreFilledByteArray(input), new TestOption[0]);
        ByteSource slice = source.slice(offset, length);
        Assert.assertArrayEquals((byte[])expected, (byte[])slice.read());
    }

    public void testCopyToStream_doesNotCloseThatStream() throws IOException {
        TestOutputStream out = new TestOutputStream(ByteStreams.nullOutputStream(), new TestOption[0]);
        ByteSourceTest.assertFalse((boolean)out.closed());
        this.source.copyTo(out);
        ByteSourceTest.assertFalse((boolean)out.closed());
    }

    public void testClosesOnErrors_copyingToByteSinkThatThrows() {
        for (TestOption option : EnumSet.of(TestOption.OPEN_THROWS, TestOption.WRITE_THROWS, TestOption.CLOSE_THROWS)) {
            TestByteSource okSource = new TestByteSource(bytes, new TestOption[0]);
            try {
                okSource.copyTo(new TestByteSink(option));
                ByteSourceTest.fail();
            }
            catch (IOException expected) {
                // empty catch block
            }
            ByteSourceTest.assertTrue((String)("stream not closed when copying to sink with option: " + (Object)((Object)option)), (!okSource.wasStreamOpened() || okSource.wasStreamClosed() ? 1 : 0) != 0);
        }
    }

    public void testClosesOnErrors_whenReadThrows() {
        TestByteSource failSource = new TestByteSource(bytes, TestOption.READ_THROWS);
        try {
            failSource.copyTo(new TestByteSink(new TestOption[0]));
            ByteSourceTest.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        ByteSourceTest.assertTrue((boolean)failSource.wasStreamClosed());
    }

    public void testClosesOnErrors_copyingToOutputStreamThatThrows() {
        TestByteSource okSource = new TestByteSource(bytes, new TestOption[0]);
        try {
            TestOutputStream out = new TestOutputStream(ByteStreams.nullOutputStream(), TestOption.WRITE_THROWS);
            okSource.copyTo(out);
            ByteSourceTest.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        ByteSourceTest.assertTrue((boolean)okSource.wasStreamClosed());
    }

    public void testConcat() throws IOException {
        ByteSource b1 = ByteSource.wrap((byte[])new byte[]{0, 1, 2, 3});
        ByteSource b2 = ByteSource.wrap((byte[])new byte[0]);
        ByteSource b3 = ByteSource.wrap((byte[])new byte[]{4, 5});
        byte[] expected = new byte[]{0, 1, 2, 3, 4, 5};
        Assert.assertArrayEquals((byte[])expected, (byte[])ByteSource.concat((Iterable)ImmutableList.of((Object)b1, (Object)b2, (Object)b3)).read());
        Assert.assertArrayEquals((byte[])expected, (byte[])ByteSource.concat((ByteSource[])new ByteSource[]{b1, b2, b3}).read());
        Assert.assertArrayEquals((byte[])expected, (byte[])ByteSource.concat((Iterator)ImmutableList.of((Object)b1, (Object)b2, (Object)b3).iterator()).read());
        ByteSourceTest.assertEquals((long)expected.length, (long)ByteSource.concat((ByteSource[])new ByteSource[]{b1, b2, b3}).size());
        ByteSourceTest.assertFalse((boolean)ByteSource.concat((ByteSource[])new ByteSource[]{b1, b2, b3}).isEmpty());
        ByteSource emptyConcat = ByteSource.concat((ByteSource[])new ByteSource[]{ByteSource.empty(), ByteSource.empty()});
        ByteSourceTest.assertTrue((boolean)emptyConcat.isEmpty());
        ByteSourceTest.assertEquals((long)0L, (long)emptyConcat.size());
    }

    public void testConcat_infiniteIterable() throws IOException {
        ByteSource source = ByteSource.wrap((byte[])new byte[]{0, 1, 2, 3});
        Iterable cycle = Iterables.cycle((Iterable)ImmutableList.of((Object)source));
        ByteSource concatenated = ByteSource.concat((Iterable)cycle);
        byte[] expected = new byte[]{0, 1, 2, 3, 0, 1, 2, 3};
        Assert.assertArrayEquals((byte[])expected, (byte[])concatenated.slice(0L, 8L).read());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testCopyExceptions() {
        if (!Closer.SuppressingSuppressor.isAvailable()) {
            TestLogHandler logHandler = new TestLogHandler();
            Closeables.logger.addHandler((Handler)logHandler);
            try {
                for (ByteSource in : BROKEN_SOURCES) {
                    ByteSourceTest.runFailureTest(in, ByteSourceTest.newNormalByteSink());
                    ByteSourceTest.assertTrue((boolean)logHandler.getStoredLogRecords().isEmpty());
                    ByteSourceTest.runFailureTest(in, BROKEN_CLOSE_SINK);
                    ByteSourceTest.assertEquals((int)(in == BROKEN_OPEN_SOURCE ? 0 : 1), (int)ByteSourceTest.getAndResetRecords(logHandler));
                }
                for (ByteSink out : BROKEN_SINKS) {
                    ByteSourceTest.runFailureTest(ByteSourceTest.newNormalByteSource(), out);
                    ByteSourceTest.assertTrue((boolean)logHandler.getStoredLogRecords().isEmpty());
                    ByteSourceTest.runFailureTest(BROKEN_CLOSE_SOURCE, out);
                    ByteSourceTest.assertEquals((int)1, (int)ByteSourceTest.getAndResetRecords(logHandler));
                }
                for (ByteSource in : BROKEN_SOURCES) {
                    for (ByteSink out : BROKEN_SINKS) {
                        ByteSourceTest.runFailureTest(in, out);
                        ByteSourceTest.assertTrue((ByteSourceTest.getAndResetRecords(logHandler) <= 1 ? 1 : 0) != 0);
                    }
                }
            }
            finally {
                Closeables.logger.removeHandler((Handler)logHandler);
            }
        } else {
            int suppressed;
            for (ByteSource in : BROKEN_SOURCES) {
                suppressed = ByteSourceTest.runSuppressionFailureTest(in, ByteSourceTest.newNormalByteSink());
                ByteSourceTest.assertEquals((int)0, (int)suppressed);
                suppressed = ByteSourceTest.runSuppressionFailureTest(in, BROKEN_CLOSE_SINK);
                ByteSourceTest.assertEquals((int)(in == BROKEN_OPEN_SOURCE ? 0 : 1), (int)suppressed);
            }
            for (ByteSink out : BROKEN_SINKS) {
                suppressed = ByteSourceTest.runSuppressionFailureTest(ByteSourceTest.newNormalByteSource(), out);
                ByteSourceTest.assertEquals((int)0, (int)suppressed);
                suppressed = ByteSourceTest.runSuppressionFailureTest(BROKEN_CLOSE_SOURCE, out);
                ByteSourceTest.assertEquals((int)1, (int)suppressed);
            }
            for (ByteSource in : BROKEN_SOURCES) {
                for (ByteSink out : BROKEN_SINKS) {
                    int suppressed2 = ByteSourceTest.runSuppressionFailureTest(in, out);
                    ByteSourceTest.assertTrue((suppressed2 <= 1 ? 1 : 0) != 0);
                }
            }
        }
    }

    private static int getAndResetRecords(TestLogHandler logHandler) {
        int records = logHandler.getStoredLogRecords().size();
        logHandler.clear();
        return records;
    }

    private static void runFailureTest(ByteSource in, ByteSink out) {
        try {
            in.copyTo(out);
            ByteSourceTest.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static int runSuppressionFailureTest(ByteSource in, ByteSink out) {
        try {
            in.copyTo(out);
            ByteSourceTest.fail();
        }
        catch (IOException expected) {
            return CloserTest.getSuppressed(expected).length;
        }
        throw new AssertionError();
    }

    private static ByteSource newNormalByteSource() {
        return ByteSource.wrap((byte[])new byte[10]);
    }

    private static ByteSink newNormalByteSink() {
        return new ByteSink(){

            public OutputStream openStream() {
                return new ByteArrayOutputStream();
            }
        };
    }
}

