/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.hash;

import com.google.common.collect.ImmutableSet;
import com.google.common.hash.BloomFilter;
import com.google.common.hash.BloomFilterStrategies;
import com.google.common.hash.Funnel;
import com.google.common.hash.Funnels;
import com.google.common.hash.HashTestUtils;
import com.google.common.hash.PrimitiveSink;
import com.google.common.math.LongMath;
import com.google.common.primitives.Ints;
import com.google.common.testing.EqualsTester;
import com.google.common.testing.NullPointerTester;
import com.google.common.testing.SerializableTester;
import java.math.RoundingMode;
import java.util.Random;
import javax.annotation.Nullable;
import junit.framework.TestCase;

public class BloomFilterTest
extends TestCase {
    public void testLargeBloomFilterDoesntOverflow() {
        long numBits = Integer.MAX_VALUE;
        BloomFilterStrategies.BitArray bitArray = new BloomFilterStrategies.BitArray(++numBits);
        BloomFilterTest.assertTrue((String)("BitArray.bitSize() must return a positive number, but was " + bitArray.bitSize()), (bitArray.bitSize() > 0L ? 1 : 0) != 0);
    }

    public void testCreateAndCheckMitz32BloomFilterWithKnownFalsePositives() {
        int i;
        int numInsertions = 1000000;
        BloomFilter bf = BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)numInsertions, (double)0.03, (BloomFilter.Strategy)BloomFilterStrategies.MURMUR128_MITZ_32);
        for (i = 0; i < numInsertions * 2; i += 2) {
            bf.put((Object)Integer.toString(i));
        }
        for (i = 0; i < numInsertions * 2; i += 2) {
            BloomFilterTest.assertTrue((boolean)bf.mightContain((Object)Integer.toString(i)));
        }
        ImmutableSet falsePositives = ImmutableSet.of((Object)49, (Object)51, (Object)59, (Object)163, (Object)199, (Object)321, (Object[])new Integer[]{325, 363, 367, 469, 545, 561, 727, 769, 773, 781});
        for (int i2 = 1; i2 < 900; i2 += 2) {
            if (falsePositives.contains((Object)i2)) continue;
            BloomFilterTest.assertFalse((String)("BF should not contain " + i2), (boolean)bf.mightContain((Object)Integer.toString(i2)));
        }
        int knownNumberOfFalsePositives = 29824;
        int numFpp = 0;
        for (int i3 = 1; i3 < numInsertions * 2; i3 += 2) {
            if (!bf.mightContain((Object)Integer.toString(i3))) continue;
            ++numFpp;
        }
        BloomFilterTest.assertEquals((int)knownNumberOfFalsePositives, (int)numFpp);
        double actualFpp = (double)knownNumberOfFalsePositives / (double)numInsertions;
        double expectedFpp = bf.expectedFpp();
        BloomFilterTest.assertEquals((double)actualFpp, (double)expectedFpp, (double)1.5E-4);
    }

    public void testCreateAndCheckBloomFilterWithKnownFalsePositives64() {
        int i;
        int numInsertions = 1000000;
        BloomFilter bf = BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)numInsertions, (double)0.03, (BloomFilter.Strategy)BloomFilterStrategies.MURMUR128_MITZ_64);
        for (i = 0; i < numInsertions * 2; i += 2) {
            bf.put((Object)Integer.toString(i));
        }
        for (i = 0; i < numInsertions * 2; i += 2) {
            BloomFilterTest.assertTrue((boolean)bf.mightContain((Object)Integer.toString(i)));
        }
        ImmutableSet falsePositives = ImmutableSet.of((Object)15, (Object)25, (Object)287, (Object)319, (Object)381, (Object)399, (Object[])new Integer[]{421, 465, 529, 697, 767, 857});
        for (int i2 = 1; i2 < 900; i2 += 2) {
            if (falsePositives.contains((Object)i2)) continue;
            BloomFilterTest.assertFalse((String)("BF should not contain " + i2), (boolean)bf.mightContain((Object)Integer.toString(i2)));
        }
        int knownNumberOfFalsePositives = 30104;
        int numFpp = 0;
        for (int i3 = 1; i3 < numInsertions * 2; i3 += 2) {
            if (!bf.mightContain((Object)Integer.toString(i3))) continue;
            ++numFpp;
        }
        BloomFilterTest.assertEquals((int)knownNumberOfFalsePositives, (int)numFpp);
        double actualFpp = (double)knownNumberOfFalsePositives / (double)numInsertions;
        double expectedFpp = bf.expectedFpp();
        BloomFilterTest.assertEquals((double)actualFpp, (double)expectedFpp, (double)3.3E-4);
    }

    public void testBasic() {
        for (double fpr = 1.0E-7; fpr < 0.1; fpr *= 10.0) {
            for (int expectedInsertions = 1; expectedInsertions <= 10000; expectedInsertions *= 10) {
                this.checkSanity((BloomFilter<Object>)BloomFilter.create(HashTestUtils.BAD_FUNNEL, (int)expectedInsertions, (double)fpr));
            }
        }
    }

    public void testPreconditions() {
        try {
            BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)-1);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)-1, (double)0.03);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)1, (double)0.0);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)1, (double)1.0);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testFailureWhenMoreThan255HashFunctionsAreNeeded() {
        try {
            int n = 1000;
            double p = 1.0E-80;
            BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)n, (double)p);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testNullPointers() {
        NullPointerTester tester = new NullPointerTester();
        tester.testAllPublicInstanceMethods((Object)BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)100));
        tester.testAllPublicStaticMethods(BloomFilter.class);
    }

    public void testOptimalHashes() {
        for (int n = 1; n < 1000; ++n) {
            for (int m = 0; m < 1000; ++m) {
                BloomFilterTest.assertTrue((BloomFilter.optimalNumOfHashFunctions((long)n, (long)m) > 0 ? 1 : 0) != 0);
            }
        }
    }

    public void testOptimalSize() {
        for (int n = 1; n < 1000; ++n) {
            for (double fpp = Double.MIN_VALUE; fpp < 1.0; fpp += 0.001) {
                BloomFilterTest.assertTrue((BloomFilter.optimalNumOfBits((long)n, (double)fpp) >= 0L ? 1 : 0) != 0);
            }
        }
        Random random = new Random(0L);
        for (int repeats = 0; repeats < 10000; ++repeats) {
            BloomFilterTest.assertTrue((BloomFilter.optimalNumOfBits((long)random.nextInt(65536), (double)random.nextDouble()) >= 0L ? 1 : 0) != 0);
        }
        BloomFilterTest.assertEquals((long)3327428144502L, (long)BloomFilter.optimalNumOfBits((long)Integer.MAX_VALUE, (double)Double.MIN_VALUE));
        try {
            BloomFilter.create(HashTestUtils.BAD_FUNNEL, (int)Integer.MAX_VALUE, (double)Double.MIN_VALUE);
            BloomFilterTest.fail((String)"we can't represent such a large BF!");
        }
        catch (IllegalArgumentException expected) {
            BloomFilterTest.assertEquals((String)"Could not create BloomFilter of 3327428144502 bits", (String)expected.getMessage());
        }
    }

    private void checkSanity(BloomFilter<Object> bf) {
        BloomFilterTest.assertFalse((boolean)bf.mightContain(new Object()));
        BloomFilterTest.assertFalse((boolean)bf.apply(new Object()));
        for (int i = 0; i < 100; ++i) {
            Object o = new Object();
            bf.put(o);
            BloomFilterTest.assertTrue((boolean)bf.mightContain(o));
            BloomFilterTest.assertTrue((boolean)bf.apply(o));
        }
    }

    public void testCopy() {
        BloomFilter original = BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)100);
        BloomFilter copy = original.copy();
        BloomFilterTest.assertNotSame((Object)original, (Object)copy);
        BloomFilterTest.assertEquals((Object)original, (Object)copy);
    }

    public void testExpectedFpp() {
        BloomFilter bf = BloomFilter.create(HashTestUtils.BAD_FUNNEL, (int)10, (double)0.03);
        double fpp = bf.expectedFpp();
        BloomFilterTest.assertEquals((Object)0.0, (Object)fpp);
        while (fpp != 1.0) {
            boolean changed = bf.put(new Object());
            double newFpp = bf.expectedFpp();
            BloomFilterTest.assertTrue((boolean)(changed ? newFpp > fpp : newFpp == fpp));
            fpp = newFpp;
        }
    }

    public void testBitSize() {
        double fpp = 0.03;
        for (int i = 1; i < 10000; ++i) {
            long numBits = BloomFilter.optimalNumOfBits((long)i, (double)fpp);
            int arraySize = Ints.checkedCast((long)LongMath.divide((long)numBits, (long)64L, (RoundingMode)RoundingMode.CEILING));
            BloomFilterTest.assertEquals((long)(arraySize * 64), (long)BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)i, (double)fpp).bitSize());
        }
    }

    public void testEquals_empty() {
        new EqualsTester().addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.byteArrayFunnel(), (int)100, (double)0.01)}).addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.byteArrayFunnel(), (int)100, (double)0.02)}).addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.byteArrayFunnel(), (int)200, (double)0.01)}).addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.byteArrayFunnel(), (int)200, (double)0.02)}).addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)100, (double)0.01)}).addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)100, (double)0.02)}).addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)200, (double)0.01)}).addEqualityGroup(new Object[]{BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)200, (double)0.02)}).testEquals();
    }

    public void testEquals() {
        BloomFilter bf1 = BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)100);
        bf1.put((Object)"1");
        bf1.put((Object)"2");
        BloomFilter bf2 = BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)100);
        bf2.put((Object)"1");
        bf2.put((Object)"2");
        new EqualsTester().addEqualityGroup(new Object[]{bf1, bf2}).testEquals();
        bf2.put((Object)"3");
        new EqualsTester().addEqualityGroup(new Object[]{bf1}).addEqualityGroup(new Object[]{bf2}).testEquals();
    }

    public void testEqualsWithCustomFunnel() {
        BloomFilter bf1 = BloomFilter.create((Funnel)new CustomFunnel(), (int)100);
        BloomFilter bf2 = BloomFilter.create((Funnel)new CustomFunnel(), (int)100);
        BloomFilterTest.assertEquals((Object)bf1, (Object)bf2);
    }

    public void testSerializationWithCustomFunnel() {
        SerializableTester.reserializeAndAssert((Object)BloomFilter.create((Funnel)new CustomFunnel(), (int)100));
    }

    public void testPutReturnValue() {
        for (int i = 0; i < 10; ++i) {
            BloomFilter bf = BloomFilter.create((Funnel)Funnels.unencodedCharsFunnel(), (int)100);
            for (int j = 0; j < 10; ++j) {
                boolean put;
                String value = new Object().toString();
                boolean mightContain = bf.mightContain((Object)value);
                BloomFilterTest.assertTrue((mightContain != (put = bf.put((Object)value)) ? 1 : 0) != 0);
            }
        }
    }

    public void testPutAll() {
        int element1 = 1;
        int element2 = 2;
        BloomFilter bf1 = BloomFilter.create((Funnel)Funnels.integerFunnel(), (int)100);
        bf1.put((Object)element1);
        BloomFilterTest.assertTrue((boolean)bf1.mightContain((Object)element1));
        BloomFilterTest.assertFalse((boolean)bf1.mightContain((Object)element2));
        BloomFilter bf2 = BloomFilter.create((Funnel)Funnels.integerFunnel(), (int)100);
        bf2.put((Object)element2);
        BloomFilterTest.assertFalse((boolean)bf2.mightContain((Object)element1));
        BloomFilterTest.assertTrue((boolean)bf2.mightContain((Object)element2));
        BloomFilterTest.assertTrue((boolean)bf1.isCompatible(bf2));
        bf1.putAll(bf2);
        BloomFilterTest.assertTrue((boolean)bf1.mightContain((Object)element1));
        BloomFilterTest.assertTrue((boolean)bf1.mightContain((Object)element2));
        BloomFilterTest.assertFalse((boolean)bf2.mightContain((Object)element1));
        BloomFilterTest.assertTrue((boolean)bf2.mightContain((Object)element2));
    }

    public void testPutAllDifferentSizes() {
        BloomFilter bf1 = BloomFilter.create((Funnel)Funnels.integerFunnel(), (int)1);
        BloomFilter bf2 = BloomFilter.create((Funnel)Funnels.integerFunnel(), (int)10);
        try {
            BloomFilterTest.assertFalse((boolean)bf1.isCompatible(bf2));
            bf1.putAll(bf2);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            BloomFilterTest.assertFalse((boolean)bf2.isCompatible(bf1));
            bf2.putAll(bf1);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testPutAllWithSelf() {
        BloomFilter bf1 = BloomFilter.create((Funnel)Funnels.integerFunnel(), (int)1);
        try {
            BloomFilterTest.assertFalse((boolean)bf1.isCompatible(bf1));
            bf1.putAll(bf1);
            BloomFilterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testJavaSerialization() {
        BloomFilter bf = BloomFilter.create((Funnel)Funnels.byteArrayFunnel(), (int)100);
        for (int i = 0; i < 10; ++i) {
            bf.put((Object)Ints.toByteArray((int)i));
        }
        BloomFilter copy = (BloomFilter)SerializableTester.reserialize((Object)bf);
        for (int i = 0; i < 10; ++i) {
            BloomFilterTest.assertTrue((boolean)copy.mightContain((Object)Ints.toByteArray((int)i)));
        }
        BloomFilterTest.assertEquals((Object)bf.expectedFpp(), (Object)copy.expectedFpp());
        SerializableTester.reserializeAndAssert((Object)bf);
    }

    public void testBloomFilterStrategies() {
        BloomFilterTest.assertEquals((int)2, (int)BloomFilterStrategies.values().length);
        BloomFilterTest.assertEquals((Object)BloomFilterStrategies.MURMUR128_MITZ_32, (Object)BloomFilterStrategies.values()[0]);
        BloomFilterTest.assertEquals((Object)BloomFilterStrategies.MURMUR128_MITZ_64, (Object)BloomFilterStrategies.values()[1]);
    }

    public void testGetDefaultStrategyFromSystemProperty() {
        System.clearProperty("com.google.common.hash.BloomFilter.useMitz32");
        BloomFilterTest.assertEquals((Object)BloomFilterStrategies.MURMUR128_MITZ_64, (Object)BloomFilter.getDefaultStrategyFromSystemProperty());
        System.setProperty("com.google.common.hash.BloomFilter.useMitz32", "true");
        BloomFilterTest.assertEquals((Object)BloomFilterStrategies.MURMUR128_MITZ_32, (Object)BloomFilter.getDefaultStrategyFromSystemProperty());
        System.setProperty("com.google.common.hash.BloomFilter.useMitz32", "TRUE");
        BloomFilterTest.assertEquals((Object)BloomFilterStrategies.MURMUR128_MITZ_32, (Object)BloomFilter.getDefaultStrategyFromSystemProperty());
        System.setProperty("com.google.common.hash.BloomFilter.useMitz32", "false");
        BloomFilterTest.assertEquals((Object)BloomFilterStrategies.MURMUR128_MITZ_64, (Object)BloomFilter.getDefaultStrategyFromSystemProperty());
    }

    private static final class CustomFunnel
    implements Funnel<Long> {
        private CustomFunnel() {
        }

        public void funnel(Long value, PrimitiveSink into) {
            into.putLong(value.longValue());
        }

        public boolean equals(@Nullable Object object) {
            return object instanceof CustomFunnel;
        }

        public int hashCode() {
            return 42;
        }
    }
}

