/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.util.concurrent;

import com.google.common.collect.Lists;
import com.google.common.testing.NullPointerTester;
import com.google.common.util.concurrent.RateLimiter;
import java.util.Arrays;
import java.util.List;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import junit.framework.TestCase;

public class RateLimiterTest
extends TestCase {
    private static final double EPSILON = 1.0E-8;
    private final FakeTicker ticker = new FakeTicker();

    public void testSimple() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)5.0);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "R0.20", "R0.20");
    }

    public void testImmediateTryAcquire() {
        RateLimiter r = RateLimiter.create((double)1.0);
        RateLimiterTest.assertTrue((String)"Unable to acquire initial permit", (boolean)r.tryAcquire());
        RateLimiterTest.assertFalse((String)"Capable of acquiring secondary permit", (boolean)r.tryAcquire());
    }

    public void testSimpleRateUpdate() {
        RateLimiter limiter = RateLimiter.create((double)5.0, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        RateLimiterTest.assertEquals((Object)5.0, (Object)limiter.getRate());
        limiter.setRate(10.0);
        RateLimiterTest.assertEquals((Object)10.0, (Object)limiter.getRate());
        try {
            limiter.setRate(0.0);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException ok) {
            // empty catch block
        }
        try {
            limiter.setRate(-10.0);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testSimpleWithWait() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)5.0);
        limiter.acquire();
        this.ticker.sleepMillis(200);
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "U0.20", "R0.00", "R0.20");
    }

    public void testSimpleAcquireReturnValues() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)5.0);
        RateLimiterTest.assertEquals((double)0.0, (double)limiter.acquire(), (double)1.0E-8);
        this.ticker.sleepMillis(200);
        RateLimiterTest.assertEquals((double)0.0, (double)limiter.acquire(), (double)1.0E-8);
        RateLimiterTest.assertEquals((double)0.2, (double)limiter.acquire(), (double)1.0E-8);
        this.assertEvents("R0.00", "U0.20", "R0.00", "R0.20");
    }

    public void testOneSecondBurst() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)5.0);
        this.ticker.sleepMillis(1000);
        this.ticker.sleepMillis(1000);
        limiter.acquire(1);
        limiter.acquire(1);
        limiter.acquire(3);
        limiter.acquire(1);
        limiter.acquire();
        this.assertEvents("U1.00", "U1.00", "R0.00", "R0.00", "R0.00", "R0.00", "R0.20");
    }

    public void testWarmUp() {
        int i;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)2.0, (long)4000L, (TimeUnit)TimeUnit.MILLISECONDS);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.ticker.sleepMillis(500);
        this.ticker.sleepMillis(4000);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.ticker.sleepMillis(500);
        this.ticker.sleepMillis(2000);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00, R1.38, R1.13, R0.88, R0.63, R0.50, R0.50, R0.50", "U0.50", "U4.00", "R0.00, R1.38, R1.13, R0.88, R0.63, R0.50, R0.50, R0.50", "U0.50", "U2.00", "R0.00, R0.50, R0.50, R0.50, R0.50, R0.50, R0.50, R0.50");
    }

    public void testWarmUpAndUpdate() {
        int i;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)2.0, (long)4000L, (TimeUnit)TimeUnit.MILLISECONDS);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.ticker.sleepMillis(4500);
        for (i = 0; i < 3; ++i) {
            limiter.acquire();
        }
        limiter.setRate(4.0);
        limiter.acquire();
        for (i = 0; i < 4; ++i) {
            limiter.acquire();
        }
        this.ticker.sleepMillis(4250);
        for (i = 0; i < 11; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00, R1.38, R1.13, R0.88, R0.63, R0.50, R0.50, R0.50", "U4.50", "R0.00, R1.38, R1.13", "R0.88", "R0.34, R0.28, R0.25, R0.25", "U4.25", "R0.00, R0.72, R0.66, R0.59, R0.53, R0.47, R0.41", "R0.34, R0.28, R0.25, R0.25");
    }

    public void testBursty() {
        RateLimiter limiter = RateLimiter.createWithCapacity((RateLimiter.SleepingTicker)this.ticker, (double)1.0, (long)10L, (TimeUnit)TimeUnit.SECONDS);
        this.ticker.sleepMillis(10000);
        limiter.acquire(11);
        limiter.acquire(1);
        limiter.acquire(1);
        this.ticker.sleepMillis(3000);
        limiter.acquire(5);
        limiter.acquire(1);
        this.assertEvents("U10.00", "R0.00", "R1.00", "R1.00", "U3.00", "R0.00", "R3.00");
    }

    public void testBurstyAndUpdate() {
        RateLimiter rateLimiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)1.0);
        rateLimiter.acquire(1);
        rateLimiter.acquire(1);
        rateLimiter.setRate(2.0);
        rateLimiter.acquire(1);
        rateLimiter.acquire(2);
        rateLimiter.acquire(4);
        rateLimiter.acquire(1);
        this.assertEvents("R0.00", "R1.00", "R1.00", "R0.50", "R1.00", "R2.00");
    }

    public void testTimeWrapping() {
        this.ticker.instant = Long.MAX_VALUE - TimeUnit.SECONDS.toNanos(1L);
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)1.0);
        for (int i = 0; i < 4; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00", "R1.00", "R1.00", "R1.00");
    }

    public void testTryGate() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)5.0);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
        this.ticker.sleepMillis(100);
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
    }

    public void testSimpleWeights() {
        RateLimiter rateLimiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)1.0);
        rateLimiter.acquire(1);
        rateLimiter.acquire(1);
        rateLimiter.acquire(2);
        rateLimiter.acquire(4);
        rateLimiter.acquire(8);
        rateLimiter.acquire(1);
        this.assertEvents("R0.00", "R1.00", "R1.00", "R2.00", "R4.00", "R8.00");
    }

    public void testInfinity_Bursty() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)Double.POSITIVE_INFINITY);
        limiter.acquire(0x1FFFFFFF);
        limiter.acquire(0x3FFFFFFF);
        limiter.acquire(Integer.MAX_VALUE);
        this.assertEvents("R0.00", "R0.00", "R0.00");
        limiter.setRate(1.0);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "R1.00", "R1.00");
        limiter.setRate(Double.POSITIVE_INFINITY);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R1.00", "R0.00", "R0.00");
    }

    public void testInfinity_WarmUp() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)Double.POSITIVE_INFINITY, (long)10L, (TimeUnit)TimeUnit.SECONDS);
        limiter.acquire(0x1FFFFFFF);
        limiter.acquire(0x3FFFFFFF);
        limiter.acquire(Integer.MAX_VALUE);
        this.assertEvents("R0.00", "R0.00", "R0.00");
        limiter.setRate(1.0);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "R1.00", "R1.00");
        limiter.setRate(Double.POSITIVE_INFINITY);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R1.00", "R0.00", "R0.00");
    }

    public void testWeNeverGetABurstMoreThanOneSec() {
        int[] rates;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)1.0);
        int[] arr$ = rates = new int[]{1000, 1, 10, 1000000, 10, 1};
        int len$ = arr$.length;
        for (int i$ = 0; i$ < len$; ++i$) {
            int rate;
            int oneSecWorthOfWork = rate = arr$[i$];
            this.ticker.sleepMillis(rate * 1000);
            limiter.setRate((double)rate);
            long burst = this.measureTotalTimeMillis(limiter, oneSecWorthOfWork, new Random());
            RateLimiterTest.assertTrue((burst <= 1000L ? 1 : 0) != 0);
            long afterBurst = this.measureTotalTimeMillis(limiter, oneSecWorthOfWork, new Random());
            RateLimiterTest.assertTrue((afterBurst >= 1000L ? 1 : 0) != 0);
        }
    }

    public void testTimeToWarmUpIsHonouredEvenWithWeights() {
        Random random = new Random();
        int maxPermits = 10;
        double[] qpsToTest = new double[]{4.0, 2.0, 1.0, 0.5, 0.1};
        for (int trial = 0; trial < 100; ++trial) {
            for (double qps : qpsToTest) {
                long warmupMillis = (long)((double)(2 * maxPermits) / qps * 1000.0);
                RateLimiter rateLimiter = RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)qps, (long)warmupMillis, (TimeUnit)TimeUnit.MILLISECONDS);
                RateLimiterTest.assertEquals((long)warmupMillis, (long)this.measureTotalTimeMillis(rateLimiter, maxPermits, random));
            }
        }
    }

    public void testNulls() {
        NullPointerTester tester = new NullPointerTester().setDefault(RateLimiter.SleepingTicker.class, (Object)this.ticker);
        tester.testStaticMethods(RateLimiter.class, NullPointerTester.Visibility.PACKAGE);
        tester.testInstanceMethods((Object)RateLimiter.create((RateLimiter.SleepingTicker)this.ticker, (double)5.0), NullPointerTester.Visibility.PACKAGE);
    }

    private long measureTotalTimeMillis(RateLimiter rateLimiter, int permits, Random random) {
        long startTime = this.ticker.instant;
        while (permits > 0) {
            int nextPermitsToAcquire = Math.max(1, random.nextInt(permits));
            permits -= nextPermitsToAcquire;
            rateLimiter.acquire(nextPermitsToAcquire);
        }
        rateLimiter.acquire(1);
        return TimeUnit.NANOSECONDS.toMillis(this.ticker.instant - startTime);
    }

    private void assertEvents(String ... events) {
        RateLimiterTest.assertEquals((String)Arrays.toString(events), (String)this.ticker.readEventsAndClear());
    }

    private static class FakeTicker
    extends RateLimiter.SleepingTicker {
        long instant = 0L;
        final List<String> events = Lists.newArrayList();

        private FakeTicker() {
        }

        public long read() {
            return this.instant;
        }

        void sleepMillis(int millis) {
            this.sleepMicros("U", TimeUnit.MILLISECONDS.toMicros(millis));
        }

        void sleepMicros(String caption, long micros) {
            this.instant += TimeUnit.MICROSECONDS.toNanos(micros);
            this.events.add(caption + String.format("%3.2f", (double)micros / 1000000.0));
        }

        void sleepMicrosUninterruptibly(long micros) {
            this.sleepMicros("R", micros);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        String readEventsAndClear() {
            try {
                String string = this.events.toString();
                return string;
            }
            finally {
                this.events.clear();
            }
        }

        public String toString() {
            return this.events.toString();
        }
    }
}

