/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.net;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.base.Ascii;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.net.InternetDomainName;
import com.google.common.testing.EqualsTester;
import com.google.common.testing.NullPointerTester;
import junit.framework.TestCase;

@GwtCompatible(emulated=true)
public final class InternetDomainNameTest
extends TestCase {
    private static final InternetDomainName UNICODE_EXAMPLE = InternetDomainName.from((String)"j\u00f8rpeland.no");
    private static final InternetDomainName PUNYCODE_EXAMPLE = InternetDomainName.from((String)"xn--jrpeland-54a.no");
    private static final String DELTA = "\u0394";
    static final String LOTS_OF_DELTAS = Strings.repeat((String)"\u0394", (int)62);
    private static final String ALMOST_TOO_MANY_LEVELS = Strings.repeat((String)"a.", (int)127);
    private static final String ALMOST_TOO_LONG = String.valueOf(Strings.repeat((String)"aaaaa.", (int)40)).concat("1234567890.c");
    private static final ImmutableSet<String> VALID_NAME = ImmutableSet.of((Object)"foo.com", (Object)"f-_-o.cOM", (Object)"f--1.com", (Object)"f11-1.com", (Object)"www", (Object)"abc.a23", (Object[])new String[]{"biz.com.ua", "x", "fOo", "f--o", "f_a", "foo.net.us\uff61ocm", "woo.com.", "a\u0394b.com", ALMOST_TOO_MANY_LEVELS, ALMOST_TOO_LONG});
    private static final ImmutableSet<String> INVALID_NAME = ImmutableSet.of((Object)"", (Object)" ", (Object)"127.0.0.1", (Object)"::1", (Object)"13", (Object)"abc.12c", (Object[])new String[]{"foo-.com", "_bar.quux", "foo+bar.com", "foo!bar.com", ".foo.com", "..bar.com", "baz..com", "..quiffle.com", "fleeb.com..", ".", "..", "...", "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa.com", "a\u0394 .com", String.valueOf(ALMOST_TOO_MANY_LEVELS).concat("com"), String.valueOf(ALMOST_TOO_LONG).concat(".c")});
    private static final ImmutableSet<String> PS = ImmutableSet.of((Object)"com", (Object)"co.uk", (Object)"foo.bd", (Object)"xxxxxx.bd", (Object)"org.mK", (Object)"us", (Object[])new String[]{"uk\uff61com.", "\u7f51\u7edc.Cn", "j\u00f8rpeland.no", "xn--jrpeland-54a.no"});
    private static final ImmutableSet<String> NO_PS = ImmutableSet.of((Object)"www", (Object)"foo.google", (Object)"x.y.z");
    private static final ImmutableSet<String> NON_PS = ImmutableSet.of((Object)"foo.bar.com", (Object)"foo.ca", (Object)"foo.bar.ca", (Object)"foo.bar.co.il", (Object)"state.CA.us", (Object)"www.state.pa.us", (Object[])new String[]{"pvt.k12.ca.us", "www.google.com", "www4.yahoo.co.uk", "home.netscape.com", "web.MIT.edu", "foo.eDu.au", "utenti.blah.IT", "dominio.com.co"});
    private static final ImmutableSet<String> TOP_PRIVATE_DOMAIN = ImmutableSet.of((Object)"google.com", (Object)"foo.Co.uk", (Object)"foo.ca.us.");
    private static final ImmutableSet<String> UNDER_PRIVATE_DOMAIN = ImmutableSet.of((Object)"foo.bar.google.com", (Object)"a.b.co.uk", (Object)"x.y.ca.us");
    private static final ImmutableSet<String> VALID_IP_ADDRS = ImmutableSet.of((Object)"1.2.3.4", (Object)"127.0.0.1", (Object)"::1", (Object)"2001:db8::1");
    private static final ImmutableSet<String> INVALID_IP_ADDRS = ImmutableSet.of((Object)"", (Object)"1", (Object)"1.2.3", (Object)"...", (Object)"1.2.3.4.5", (Object)"400.500.600.700", (Object[])new String[]{":", ":::1", "2001:db8:"});
    private static final ImmutableSet<String> SOMEWHERE_UNDER_PS = ImmutableSet.of((Object)"foo.bar.google.com", (Object)"a.b.c.1.2.3.ca.us", (Object)"site.jp", (Object)"uomi-online.kir.jp", (Object)"jprs.co.jp", (Object)"site.quick.jp", (Object[])new String[]{"site.tenki.jp", "site.or.jp", "site.gr.jp", "site.ne.jp", "site.ac.jp", "site.ad.jp", "site.ed.jp", "site.geo.jp", "site.go.jp", "site.lg.jp", "1.fm", "site.cc", "site.ee", "site.fi", "site.fm", "site.gr", "www.leguide.ma", "site.ma", "some.org.mk", "site.mk", "site.tv", "site.us", "www.odev.us", "www.GOOGLE.com", "www.com", "google.com", "www7.google.co.uk", "google.Co.uK", "jobs.kt.com.", "home.netscape.com", "web.stanford.edu", "stanford.edu", "state.ca.us", "www.state.ca.us", "state.ca.us", "pvt.k12.ca.us", "www.rave.ca.", "cnn.ca", "ledger-enquirer.com", "it-trace.ch", "cool.dk", "cool.co.uk", "cool.de", "cool.es", "cool\uff61fr", "cool.nl", "members.blah.nl.", "cool.se", "utenti.blah.it", "kt.co", "a\u7f51\u7edcA.\u7f51\u7edc.Cn"});

    public void testValid() {
        for (String name : VALID_NAME) {
            InternetDomainName.from((String)name);
        }
    }

    public void testInvalid() {
        for (String name : INVALID_NAME) {
            try {
                InternetDomainName.from((String)name);
                String string = String.valueOf(String.valueOf(name));
                InternetDomainNameTest.fail((String)new StringBuilder(28 + string.length()).append("Should have been invalid: '").append(string).append("'").toString());
            }
            catch (IllegalArgumentException illegalArgumentException) {}
        }
    }

    public void testPublicSuffix() {
        InternetDomainName domain;
        for (String name : PS) {
            domain = InternetDomainName.from((String)name);
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.isPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.hasPublicSuffix());
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isUnderPublicSuffix());
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isTopPrivateDomain());
            InternetDomainNameTest.assertEquals((Object)domain, (Object)domain.publicSuffix());
        }
        for (String name : NO_PS) {
            domain = InternetDomainName.from((String)name);
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isPublicSuffix());
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.hasPublicSuffix());
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isUnderPublicSuffix());
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isTopPrivateDomain());
            InternetDomainNameTest.assertNull((Object)domain.publicSuffix());
        }
        for (String name : NON_PS) {
            domain = InternetDomainName.from((String)name);
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.hasPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.isUnderPublicSuffix());
        }
    }

    public void testUnderPublicSuffix() {
        for (String name : SOMEWHERE_UNDER_PS) {
            InternetDomainName domain = InternetDomainName.from((String)name);
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.hasPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.isUnderPublicSuffix());
        }
    }

    public void testTopPrivateDomain() {
        for (String name : TOP_PRIVATE_DOMAIN) {
            InternetDomainName domain = InternetDomainName.from((String)name);
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.hasPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.isUnderPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.isTopPrivateDomain());
            InternetDomainNameTest.assertEquals((Object)domain.parent(), (Object)domain.publicSuffix());
        }
    }

    public void testUnderPrivateDomain() {
        for (String name : UNDER_PRIVATE_DOMAIN) {
            InternetDomainName domain = InternetDomainName.from((String)name);
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.hasPublicSuffix());
            InternetDomainNameTest.assertTrue((String)name, (boolean)domain.isUnderPublicSuffix());
            InternetDomainNameTest.assertFalse((String)name, (boolean)domain.isTopPrivateDomain());
        }
    }

    public void testParent() {
        InternetDomainNameTest.assertEquals((String)"com", (String)InternetDomainName.from((String)"google.com").parent().toString());
        InternetDomainNameTest.assertEquals((String)"uk", (String)InternetDomainName.from((String)"co.uk").parent().toString());
        InternetDomainNameTest.assertEquals((String)"google.com", (String)InternetDomainName.from((String)"www.google.com").parent().toString());
        try {
            InternetDomainName.from((String)"com").parent();
            InternetDomainNameTest.fail((String)"'com' should throw ISE on .parent() call");
        }
        catch (IllegalStateException illegalStateException) {
            // empty catch block
        }
    }

    public void testChild() {
        InternetDomainName domain = InternetDomainName.from((String)"foo.com");
        InternetDomainNameTest.assertEquals((String)"www.foo.com", (String)domain.child("www").toString());
        try {
            domain.child("www.");
            InternetDomainNameTest.fail((String)"www..google.com should have been invalid");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testParentChild() {
        InternetDomainName origin = InternetDomainName.from((String)"foo.com");
        InternetDomainName parent = origin.parent();
        InternetDomainNameTest.assertEquals((String)"com", (String)parent.toString());
        InternetDomainName child = parent.child(LOTS_OF_DELTAS);
        child.child(LOTS_OF_DELTAS);
    }

    public void testValidTopPrivateDomain() {
        InternetDomainName googleDomain = InternetDomainName.from((String)"google.com");
        InternetDomainNameTest.assertEquals((Object)googleDomain, (Object)googleDomain.topPrivateDomain());
        InternetDomainNameTest.assertEquals((Object)googleDomain, (Object)googleDomain.child("mail").topPrivateDomain());
        InternetDomainNameTest.assertEquals((Object)googleDomain, (Object)googleDomain.child("foo.bar").topPrivateDomain());
    }

    public void testInvalidTopPrivateDomain() {
        ImmutableSet badCookieDomains = ImmutableSet.of((Object)"co.uk", (Object)"foo", (Object)"com");
        for (String domain : badCookieDomains) {
            try {
                InternetDomainName.from((String)domain).topPrivateDomain();
                InternetDomainNameTest.fail((String)domain);
            }
            catch (IllegalStateException expected) {}
        }
    }

    public void testIsValid() {
        Iterable validCases = Iterables.concat(VALID_NAME, PS, NO_PS, NON_PS);
        Iterable invalidCases = Iterables.concat(INVALID_NAME, VALID_IP_ADDRS, INVALID_IP_ADDRS);
        for (String valid : validCases) {
            InternetDomainNameTest.assertTrue((String)valid, (boolean)InternetDomainName.isValid((String)valid));
        }
        for (String invalid : invalidCases) {
            InternetDomainNameTest.assertFalse((String)invalid, (boolean)InternetDomainName.isValid((String)invalid));
        }
    }

    public void testToString() {
        for (String inputName : SOMEWHERE_UNDER_PS) {
            InternetDomainName domain = InternetDomainName.from((String)inputName);
            String expectedName = Ascii.toLowerCase((String)inputName);
            if ((expectedName = expectedName.replaceAll("[\u3002\uff0e\uff61]", ".")).endsWith(".")) {
                expectedName = expectedName.substring(0, expectedName.length() - 1);
            }
            InternetDomainNameTest.assertEquals((String)expectedName, (String)domain.toString());
        }
    }

    public void testExclusion() {
        InternetDomainName domain = InternetDomainName.from((String)"foo.teledata.mz");
        InternetDomainNameTest.assertTrue((boolean)domain.hasPublicSuffix());
        InternetDomainNameTest.assertEquals((String)"mz", (String)domain.publicSuffix().toString());
        InternetDomainNameTest.assertFalse((boolean)domain.publicSuffix().isPublicSuffix());
    }

    public void testMultipleUnders() {
        InternetDomainName domain = InternetDomainName.from((String)"www.essex.sch.uk");
        InternetDomainNameTest.assertTrue((boolean)domain.hasPublicSuffix());
        InternetDomainNameTest.assertEquals((String)"essex.sch.uk", (String)domain.publicSuffix().toString());
        InternetDomainNameTest.assertEquals((String)"www.essex.sch.uk", (String)domain.topPrivateDomain().toString());
    }

    public void testEquality() {
        new EqualsTester().addEqualityGroup(new Object[]{InternetDomainNameTest.idn("google.com"), InternetDomainNameTest.idn("google.com"), InternetDomainNameTest.idn("GOOGLE.COM")}).addEqualityGroup(new Object[]{InternetDomainNameTest.idn("www.google.com")}).addEqualityGroup(new Object[]{UNICODE_EXAMPLE}).addEqualityGroup(new Object[]{PUNYCODE_EXAMPLE}).testEquals();
    }

    private static InternetDomainName idn(String domain) {
        return InternetDomainName.from((String)domain);
    }

    @GwtIncompatible(value="NullPointerTester")
    public void testNulls() {
        NullPointerTester tester = new NullPointerTester();
        tester.testAllPublicStaticMethods(InternetDomainName.class);
        tester.testAllPublicInstanceMethods((Object)InternetDomainName.from((String)"google.com"));
    }
}

