/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.graph;

import com.google.common.collect.ImmutableSet;
import com.google.common.graph.EndpointPair;
import com.google.common.graph.Graph;
import com.google.common.graph.Graphs;
import com.google.common.graph.ImmutableGraph;
import com.google.common.graph.MutableGraph;
import com.google.common.graph.TestUtil;
import com.google.common.truth.Truth;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

public abstract class AbstractGraphTest {
    MutableGraph<Integer> graph;
    static final Integer N1 = 1;
    static final Integer N2 = 2;
    static final Integer N3 = 3;
    static final Integer N4 = 4;
    static final Integer N5 = 5;
    static final Integer NODE_NOT_IN_GRAPH = 1000;
    static final String ERROR_ELEMENT_NOT_IN_GRAPH = "not an element of this graph";
    static final String NODE_STRING = "Node";
    static final String ERROR_MODIFIABLE_SET = "Set returned is unexpectedly modifiable";
    static final String ERROR_SELF_LOOP = "self-loops are not allowed";
    static final String ERROR_NODE_NOT_IN_GRAPH = "Should not be allowed to pass a node that is not an element of the graph.";
    static final String ERROR_ADDED_SELF_LOOP = "Should not be allowed to add a self-loop edge.";

    public abstract MutableGraph<Integer> createGraph();

    @CanIgnoreReturnValue
    protected boolean addNode(Integer n) {
        return this.graph.addNode((Object)n);
    }

    @CanIgnoreReturnValue
    protected boolean putEdge(Integer n1, Integer n2) {
        this.graph.addNode((Object)n1);
        this.graph.addNode((Object)n2);
        return this.graph.putEdge((Object)n1, (Object)n2);
    }

    @Before
    public void init() {
        this.graph = this.createGraph();
    }

    @After
    public void validateGraphState() {
        AbstractGraphTest.validateGraph(this.graph);
    }

    static <N> void validateGraph(Graph<N> graph) {
        TestUtil.assertStronglyEquivalent(graph, Graphs.copyOf(graph));
        TestUtil.assertStronglyEquivalent(graph, ImmutableGraph.copyOf(graph));
        String graphString = graph.toString();
        Truth.assertThat((String)graphString).contains((CharSequence)("isDirected: " + graph.isDirected()));
        Truth.assertThat((String)graphString).contains((CharSequence)("allowsSelfLoops: " + graph.allowsSelfLoops()));
        int nodeStart = graphString.indexOf("nodes:");
        int edgeStart = graphString.indexOf("edges:");
        String nodeString = graphString.substring(nodeStart, edgeStart);
        HashSet<EndpointPair> allEndpointPairs = new HashSet<EndpointPair>();
        for (Object node : TestUtil.sanityCheckSet(graph.nodes())) {
            Truth.assertThat((String)nodeString).contains((CharSequence)node.toString());
            if (graph.isDirected()) {
                Truth.assertThat((Integer)graph.degree(node)).isEqualTo((Object)(graph.inDegree(node) + graph.outDegree(node)));
                Truth.assertThat((Iterable)graph.predecessors(node)).hasSize(graph.inDegree(node));
                Truth.assertThat((Iterable)graph.successors(node)).hasSize(graph.outDegree(node));
            } else {
                int selfLoopCount = graph.adjacentNodes(node).contains(node) ? 1 : 0;
                Truth.assertThat((Integer)graph.degree(node)).isEqualTo((Object)(graph.adjacentNodes(node).size() + selfLoopCount));
                Truth.assertThat((Iterable)graph.predecessors(node)).isEqualTo((Object)graph.adjacentNodes(node));
                Truth.assertThat((Iterable)graph.successors(node)).isEqualTo((Object)graph.adjacentNodes(node));
                Truth.assertThat((Integer)graph.inDegree(node)).isEqualTo((Object)graph.degree(node));
                Truth.assertThat((Integer)graph.outDegree(node)).isEqualTo((Object)graph.degree(node));
            }
            for (Object adjacentNode : TestUtil.sanityCheckSet(graph.adjacentNodes(node))) {
                if (!graph.allowsSelfLoops()) {
                    Truth.assertThat(node).isNotEqualTo(adjacentNode);
                }
                Truth.assertThat((Boolean)(graph.predecessors(node).contains(adjacentNode) || graph.successors(node).contains(adjacentNode) ? 1 : 0)).isTrue();
            }
            for (Object predecessor : TestUtil.sanityCheckSet(graph.predecessors(node))) {
                Truth.assertThat((Iterable)graph.successors(predecessor)).contains(node);
            }
            for (Object successor : TestUtil.sanityCheckSet(graph.successors(node))) {
                allEndpointPairs.add(EndpointPair.of(graph, node, successor));
                Truth.assertThat((Iterable)graph.predecessors(successor)).contains(node);
            }
        }
        TestUtil.sanityCheckSet(graph.edges());
        Truth.assertThat((Iterable)graph.edges()).doesNotContain((Object)EndpointPair.of(graph, (Object)new Object(), (Object)new Object()));
        Truth.assertThat((Iterable)graph.edges()).isEqualTo(allEndpointPairs);
    }

    @Test
    public abstract void nodes_checkReturnedSetMutability();

    @Test
    public abstract void adjacentNodes_checkReturnedSetMutability();

    @Test
    public abstract void predecessors_checkReturnedSetMutability();

    @Test
    public abstract void successors_checkReturnedSetMutability();

    @Test
    public void nodes_oneNode() {
        this.addNode(N1);
        Truth.assertThat((Iterable)this.graph.nodes()).containsExactly(new Object[]{N1});
    }

    @Test
    public void nodes_noNodes() {
        Truth.assertThat((Iterable)this.graph.nodes()).isEmpty();
    }

    @Test
    public void adjacentNodes_oneEdge() {
        this.putEdge(N1, N2);
        Truth.assertThat((Iterable)this.graph.adjacentNodes((Object)N1)).containsExactly(new Object[]{N2});
        Truth.assertThat((Iterable)this.graph.adjacentNodes((Object)N2)).containsExactly(new Object[]{N1});
    }

    @Test
    public void adjacentNodes_noAdjacentNodes() {
        this.addNode(N1);
        Truth.assertThat((Iterable)this.graph.adjacentNodes((Object)N1)).isEmpty();
    }

    @Test
    public void adjacentNodes_nodeNotInGraph() {
        try {
            this.graph.adjacentNodes((Object)NODE_NOT_IN_GRAPH);
            Assert.fail((String)ERROR_NODE_NOT_IN_GRAPH);
        }
        catch (IllegalArgumentException e) {
            AbstractGraphTest.assertNodeNotInGraphErrorMessage(e);
        }
    }

    @Test
    public void predecessors_noPredecessors() {
        this.addNode(N1);
        Truth.assertThat((Iterable)this.graph.predecessors((Object)N1)).isEmpty();
    }

    @Test
    public void predecessors_nodeNotInGraph() {
        try {
            this.graph.predecessors((Object)NODE_NOT_IN_GRAPH);
            Assert.fail((String)ERROR_NODE_NOT_IN_GRAPH);
        }
        catch (IllegalArgumentException e) {
            AbstractGraphTest.assertNodeNotInGraphErrorMessage(e);
        }
    }

    @Test
    public void successors_noSuccessors() {
        this.addNode(N1);
        Truth.assertThat((Iterable)this.graph.successors((Object)N1)).isEmpty();
    }

    @Test
    public void successors_nodeNotInGraph() {
        try {
            this.graph.successors((Object)NODE_NOT_IN_GRAPH);
            Assert.fail((String)ERROR_NODE_NOT_IN_GRAPH);
        }
        catch (IllegalArgumentException e) {
            AbstractGraphTest.assertNodeNotInGraphErrorMessage(e);
        }
    }

    @Test
    public void degree_oneEdge() {
        this.putEdge(N1, N2);
        Truth.assertThat((Integer)this.graph.degree((Object)N1)).isEqualTo((Object)1);
        Truth.assertThat((Integer)this.graph.degree((Object)N2)).isEqualTo((Object)1);
    }

    @Test
    public void degree_isolatedNode() {
        this.addNode(N1);
        Truth.assertThat((Integer)this.graph.degree((Object)N1)).isEqualTo((Object)0);
    }

    @Test
    public void degree_nodeNotInGraph() {
        try {
            this.graph.degree((Object)NODE_NOT_IN_GRAPH);
            Assert.fail((String)ERROR_NODE_NOT_IN_GRAPH);
        }
        catch (IllegalArgumentException e) {
            AbstractGraphTest.assertNodeNotInGraphErrorMessage(e);
        }
    }

    @Test
    public void inDegree_isolatedNode() {
        this.addNode(N1);
        Truth.assertThat((Integer)this.graph.inDegree((Object)N1)).isEqualTo((Object)0);
    }

    @Test
    public void inDegree_nodeNotInGraph() {
        try {
            this.graph.inDegree((Object)NODE_NOT_IN_GRAPH);
            Assert.fail((String)ERROR_NODE_NOT_IN_GRAPH);
        }
        catch (IllegalArgumentException e) {
            AbstractGraphTest.assertNodeNotInGraphErrorMessage(e);
        }
    }

    @Test
    public void outDegree_isolatedNode() {
        this.addNode(N1);
        Truth.assertThat((Integer)this.graph.outDegree((Object)N1)).isEqualTo((Object)0);
    }

    @Test
    public void outDegree_nodeNotInGraph() {
        try {
            this.graph.outDegree((Object)NODE_NOT_IN_GRAPH);
            Assert.fail((String)ERROR_NODE_NOT_IN_GRAPH);
        }
        catch (IllegalArgumentException e) {
            AbstractGraphTest.assertNodeNotInGraphErrorMessage(e);
        }
    }

    @Test
    public void addNode_newNode() {
        Truth.assertThat((Boolean)this.addNode(N1)).isTrue();
        Truth.assertThat((Iterable)this.graph.nodes()).contains((Object)N1);
    }

    @Test
    public void addNode_existingNode() {
        this.addNode(N1);
        ImmutableSet nodes = ImmutableSet.copyOf((Collection)this.graph.nodes());
        Truth.assertThat((Boolean)this.addNode(N1)).isFalse();
        Truth.assertThat((Iterable)this.graph.nodes()).containsExactlyElementsIn((Iterable)nodes);
    }

    @Test
    public void removeNode_existingNode() {
        this.putEdge(N1, N2);
        this.putEdge(N4, N1);
        Truth.assertThat((Boolean)this.graph.removeNode((Object)N1)).isTrue();
        Truth.assertThat((Boolean)this.graph.removeNode((Object)N1)).isFalse();
        Truth.assertThat((Iterable)this.graph.nodes()).containsExactly(new Object[]{N2, N4});
        Truth.assertThat((Iterable)this.graph.adjacentNodes((Object)N2)).isEmpty();
        Truth.assertThat((Iterable)this.graph.adjacentNodes((Object)N4)).isEmpty();
    }

    @Test
    public void removeNode_antiparallelEdges() {
        this.putEdge(N1, N2);
        this.putEdge(N2, N1);
        Truth.assertThat((Boolean)this.graph.removeNode((Object)N1)).isTrue();
        Truth.assertThat((Iterable)this.graph.nodes()).containsExactly(new Object[]{N2});
        Truth.assertThat((Iterable)this.graph.edges()).isEmpty();
        Truth.assertThat((Boolean)this.graph.removeNode((Object)N2)).isTrue();
        Truth.assertThat((Iterable)this.graph.nodes()).isEmpty();
        Truth.assertThat((Iterable)this.graph.edges()).isEmpty();
    }

    @Test
    public void removeNode_nodeNotPresent() {
        this.addNode(N1);
        ImmutableSet nodes = ImmutableSet.copyOf((Collection)this.graph.nodes());
        Truth.assertThat((Boolean)this.graph.removeNode((Object)NODE_NOT_IN_GRAPH)).isFalse();
        Truth.assertThat((Iterable)this.graph.nodes()).containsExactlyElementsIn((Iterable)nodes);
    }

    @Test
    public void removeNode_queryAfterRemoval() {
        this.addNode(N1);
        Set unused = this.graph.adjacentNodes((Object)N1);
        Truth.assertThat((Boolean)this.graph.removeNode((Object)N1)).isTrue();
        try {
            this.graph.adjacentNodes((Object)N1);
            Assert.fail((String)ERROR_NODE_NOT_IN_GRAPH);
        }
        catch (IllegalArgumentException e) {
            AbstractGraphTest.assertNodeNotInGraphErrorMessage(e);
        }
    }

    @Test
    public void removeEdge_existingEdge() {
        this.putEdge(N1, N2);
        Truth.assertThat((Iterable)this.graph.successors((Object)N1)).containsExactly(new Object[]{N2});
        Truth.assertThat((Iterable)this.graph.predecessors((Object)N2)).containsExactly(new Object[]{N1});
        Truth.assertThat((Boolean)this.graph.removeEdge((Object)N1, (Object)N2)).isTrue();
        Truth.assertThat((Boolean)this.graph.removeEdge((Object)N1, (Object)N2)).isFalse();
        Truth.assertThat((Iterable)this.graph.successors((Object)N1)).isEmpty();
        Truth.assertThat((Iterable)this.graph.predecessors((Object)N2)).isEmpty();
    }

    @Test
    public void removeEdge_oneOfMany() {
        this.putEdge(N1, N2);
        this.putEdge(N1, N3);
        this.putEdge(N1, N4);
        Truth.assertThat((Boolean)this.graph.removeEdge((Object)N1, (Object)N3)).isTrue();
        Truth.assertThat((Iterable)this.graph.adjacentNodes((Object)N1)).containsExactly(new Object[]{N2, N4});
    }

    @Test
    public void removeEdge_nodeNotPresent() {
        this.putEdge(N1, N2);
        Truth.assertThat((Boolean)this.graph.removeEdge((Object)N1, (Object)NODE_NOT_IN_GRAPH)).isFalse();
        Truth.assertThat((Iterable)this.graph.successors((Object)N1)).contains((Object)N2);
    }

    @Test
    public void removeEdge_edgeNotPresent() {
        this.putEdge(N1, N2);
        this.addNode(N3);
        Truth.assertThat((Boolean)this.graph.removeEdge((Object)N1, (Object)N3)).isFalse();
        Truth.assertThat((Iterable)this.graph.successors((Object)N1)).contains((Object)N2);
    }

    static void assertNodeNotInGraphErrorMessage(Throwable throwable) {
        Truth.assertThat((String)throwable.getMessage()).startsWith(NODE_STRING);
        Truth.assertThat((String)throwable.getMessage()).contains((CharSequence)ERROR_ELEMENT_NOT_IN_GRAPH);
    }
}

