/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.util.concurrent;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.testing.NullPointerTester;
import com.google.common.testing.TearDown;
import com.google.common.testing.TearDownAccepter;
import com.google.common.testing.TearDownStack;
import com.google.common.util.concurrent.InterruptionUtil;
import com.google.common.util.concurrent.Uninterruptibles;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import junit.framework.Assert;
import junit.framework.TestCase;

public class UninterruptiblesTest
extends TestCase {
    private static final String EXPECTED_TAKE = "expectedTake";
    private static final long LONG_DELAY_MS = 2500L;
    private static final long SLEEP_SLACK = 2L;
    private final TearDownStack tearDownStack = new TearDownStack();

    protected void setUp() {
        if (Thread.currentThread().isInterrupted()) {
            throw new AssertionError((Object)"Thread interrupted on test entry. Some test probably didn't clear the interrupt state");
        }
        this.tearDownStack.addTearDown(new TearDown(){

            public void tearDown() {
                Thread.interrupted();
            }
        });
    }

    protected void tearDown() {
        this.tearDownStack.runTearDown();
    }

    public void testNull() throws Exception {
        new NullPointerTester().setDefault(CountDownLatch.class, (Object)new CountDownLatch(0)).setDefault(Semaphore.class, (Object)new Semaphore(999)).testAllPublicStaticMethods(Uninterruptibles.class);
    }

    public void testPutWithNoWait() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        ArrayBlockingQueue queue = new ArrayBlockingQueue(999);
        Uninterruptibles.putUninterruptibly(queue, (Object)"");
        UninterruptiblesTest.assertTimeNotPassed(stopwatch, 2500L);
        UninterruptiblesTest.assertEquals((String)"", (String)((String)queue.peek()));
    }

    public void testPutNoInterrupt() {
        TimedPutQueue queue = TimedPutQueue.createWithDelay(20L);
        queue.putSuccessfully();
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testPutSingleInterrupt() {
        TimedPutQueue queue = TimedPutQueue.createWithDelay(50L);
        UninterruptiblesTest.requestInterruptIn(10L);
        queue.putSuccessfully();
        UninterruptiblesTest.assertInterrupted();
    }

    public void testPutMultiInterrupt() {
        TimedPutQueue queue = TimedPutQueue.createWithDelay(100L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        queue.putSuccessfully();
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTakeWithNoWait() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        ArrayBlockingQueue<String> queue = new ArrayBlockingQueue<String>(1);
        UninterruptiblesTest.assertTrue((boolean)queue.offer(""));
        UninterruptiblesTest.assertEquals((String)"", (String)((String)Uninterruptibles.takeUninterruptibly(queue)));
        UninterruptiblesTest.assertTimeNotPassed(stopwatch, 2500L);
    }

    public void testTakeNoInterrupt() {
        TimedTakeQueue queue = TimedTakeQueue.createWithDelay(20L);
        queue.takeSuccessfully();
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testTakeSingleInterrupt() {
        TimedTakeQueue queue = TimedTakeQueue.createWithDelay(50L);
        UninterruptiblesTest.requestInterruptIn(10L);
        queue.takeSuccessfully();
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTakeMultiInterrupt() {
        TimedTakeQueue queue = TimedTakeQueue.createWithDelay(100L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        queue.takeSuccessfully();
        UninterruptiblesTest.assertInterrupted();
    }

    public void testJoinWithNoWait() throws InterruptedException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        Thread thread = new Thread(new JoinTarget(15L));
        thread.start();
        thread.join();
        UninterruptiblesTest.assertFalse((boolean)thread.isAlive());
        Uninterruptibles.joinUninterruptibly((Thread)thread);
        Uninterruptibles.joinUninterruptibly((Thread)thread, (long)0L, (TimeUnit)TimeUnit.MILLISECONDS);
        Uninterruptibles.joinUninterruptibly((Thread)thread, (long)-42L, (TimeUnit)TimeUnit.MILLISECONDS);
        Uninterruptibles.joinUninterruptibly((Thread)thread, (long)2500L, (TimeUnit)TimeUnit.MILLISECONDS);
        UninterruptiblesTest.assertTimeNotPassed(stopwatch, 2500L);
    }

    public void testJoinNoInterrupt() {
        TimedThread thread = TimedThread.createWithDelay(20L);
        thread.joinSuccessfully();
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testJoinTimeoutNoInterruptNotExpired() {
        TimedThread thread = TimedThread.createWithDelay(20L);
        thread.joinSuccessfully(2500L);
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testJoinTimeoutNoInterruptExpired() {
        TimedThread thread = TimedThread.createWithDelay(2500L);
        thread.joinUnsuccessfully(30L);
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testJoinSingleInterrupt() {
        TimedThread thread = TimedThread.createWithDelay(50L);
        UninterruptiblesTest.requestInterruptIn(10L);
        thread.joinSuccessfully();
        UninterruptiblesTest.assertInterrupted();
    }

    public void testJoinTimeoutSingleInterruptNoExpire() {
        TimedThread thread = TimedThread.createWithDelay(50L);
        UninterruptiblesTest.requestInterruptIn(10L);
        thread.joinSuccessfully(2500L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testJoinTimeoutSingleInterruptExpired() {
        TimedThread thread = TimedThread.createWithDelay(2500L);
        UninterruptiblesTest.requestInterruptIn(10L);
        thread.joinUnsuccessfully(50L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testJoinMultiInterrupt() {
        TimedThread thread = TimedThread.createWithDelay(100L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        thread.joinSuccessfully();
        UninterruptiblesTest.assertInterrupted();
    }

    public void testJoinTimeoutMultiInterruptNoExpire() {
        TimedThread thread = TimedThread.createWithDelay(100L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        thread.joinSuccessfully(2500L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testJoinTimeoutMultiInterruptExpired() {
        TimedThread thread = TimedThread.createWithDelay(2500L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        thread.joinUnsuccessfully(70L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testSleepNoInterrupt() {
        UninterruptiblesTest.sleepSuccessfully(10L);
    }

    public void testSleepSingleInterrupt() {
        UninterruptiblesTest.requestInterruptIn(10L);
        UninterruptiblesTest.sleepSuccessfully(50L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testSleepMultiInterrupt() {
        InterruptionUtil.repeatedlyInterruptTestThread(10L, (TearDownAccepter)this.tearDownStack);
        UninterruptiblesTest.sleepSuccessfully(100L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireWithNoWait() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        Semaphore semaphore = new Semaphore(99);
        UninterruptiblesTest.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)semaphore, (long)0L, (TimeUnit)TimeUnit.MILLISECONDS));
        UninterruptiblesTest.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)semaphore, (long)-42L, (TimeUnit)TimeUnit.MILLISECONDS));
        UninterruptiblesTest.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)semaphore, (long)2500L, (TimeUnit)TimeUnit.MILLISECONDS));
        UninterruptiblesTest.assertTimeNotPassed(stopwatch, 2500L);
    }

    public void testTryAcquireTimeoutNoInterruptNotExpired() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(20L);
        semaphore.tryAcquireSuccessfully(2500L);
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testTryAcquireTimeoutNoInterruptExpired() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(2500L);
        semaphore.tryAcquireUnsuccessfully(30L);
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testTryAcquireTimeoutSingleInterruptNoExpire() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(50L);
        UninterruptiblesTest.requestInterruptIn(10L);
        semaphore.tryAcquireSuccessfully(2500L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireTimeoutSingleInterruptExpired() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(2500L);
        UninterruptiblesTest.requestInterruptIn(10L);
        semaphore.tryAcquireUnsuccessfully(50L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireTimeoutMultiInterruptNoExpire() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(100L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        semaphore.tryAcquireSuccessfully(2500L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireTimeoutMultiInterruptExpired() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(2500L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        semaphore.tryAcquireUnsuccessfully(70L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireWithNoWaitMultiPermit() {
        Stopwatch stopwatch = Stopwatch.createStarted();
        Semaphore semaphore = new Semaphore(99);
        UninterruptiblesTest.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)semaphore, (int)10, (long)0L, (TimeUnit)TimeUnit.MILLISECONDS));
        UninterruptiblesTest.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)semaphore, (int)10, (long)-42L, (TimeUnit)TimeUnit.MILLISECONDS));
        UninterruptiblesTest.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)semaphore, (int)10, (long)2500L, (TimeUnit)TimeUnit.MILLISECONDS));
        UninterruptiblesTest.assertTimeNotPassed(stopwatch, 2500L);
    }

    public void testTryAcquireTimeoutNoInterruptNotExpiredMultiPermit() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(20L);
        semaphore.tryAcquireSuccessfully(10, 2500L);
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testTryAcquireTimeoutNoInterruptExpiredMultiPermit() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(2500L);
        semaphore.tryAcquireUnsuccessfully(10, 30L);
        UninterruptiblesTest.assertNotInterrupted();
    }

    public void testTryAcquireTimeoutSingleInterruptNoExpireMultiPermit() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(50L);
        UninterruptiblesTest.requestInterruptIn(10L);
        semaphore.tryAcquireSuccessfully(10, 2500L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireTimeoutSingleInterruptExpiredMultiPermit() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(2500L);
        UninterruptiblesTest.requestInterruptIn(10L);
        semaphore.tryAcquireUnsuccessfully(10, 50L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireTimeoutMultiInterruptNoExpireMultiPermit() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(100L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        semaphore.tryAcquireSuccessfully(10, 2500L);
        UninterruptiblesTest.assertInterrupted();
    }

    public void testTryAcquireTimeoutMultiInterruptExpiredMultiPermit() {
        TimedSemaphore semaphore = TimedSemaphore.createWithDelay(2500L);
        InterruptionUtil.repeatedlyInterruptTestThread(20L, (TearDownAccepter)this.tearDownStack);
        semaphore.tryAcquireUnsuccessfully(10, 70L);
        UninterruptiblesTest.assertInterrupted();
    }

    private static void sleepSuccessfully(long sleepMillis) {
        Completion completed = new Completion(sleepMillis - 2L);
        Uninterruptibles.sleepUninterruptibly((long)sleepMillis, (TimeUnit)TimeUnit.MILLISECONDS);
        completed.assertCompletionExpected();
    }

    private static void assertTimeNotPassed(Stopwatch stopwatch, long timelimitMillis) {
        long elapsedMillis = stopwatch.elapsed(TimeUnit.MILLISECONDS);
        UninterruptiblesTest.assertTrue((elapsedMillis < timelimitMillis ? 1 : 0) != 0);
    }

    private static void assertInterrupted() {
        try {
            Thread.sleep(2500L);
            UninterruptiblesTest.fail((String)"Dude, where's my interrupt?");
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private static void assertNotInterrupted() {
        UninterruptiblesTest.assertFalse((boolean)Thread.interrupted());
    }

    private static void requestInterruptIn(long millis) {
        InterruptionUtil.requestInterruptIn(millis, TimeUnit.MILLISECONDS);
    }

    private static class Release
    extends DelayedActionRunnable {
        private final Semaphore semaphore;

        public Release(Semaphore semaphore, long tMinus) {
            super(tMinus);
            this.semaphore = semaphore;
        }

        @Override
        protected void doAction() {
            this.semaphore.release(10);
        }
    }

    private static class JoinTarget
    extends DelayedActionRunnable {
        public JoinTarget(long tMinus) {
            super(tMinus);
        }

        @Override
        protected void doAction() {
        }
    }

    private static final class TimedThread {
        private final Thread thread;
        private final Completion completed;

        static TimedThread createWithDelay(long countdownInMillis) {
            return new TimedThread(countdownInMillis);
        }

        private TimedThread(long expectedCompletionWaitMillis) {
            this.completed = new Completion(expectedCompletionWaitMillis);
            this.thread = new Thread(new JoinTarget(expectedCompletionWaitMillis));
            this.thread.start();
        }

        void joinSuccessfully() {
            Uninterruptibles.joinUninterruptibly((Thread)this.thread);
            this.completed.assertCompletionExpected();
            Assert.assertEquals((Object)((Object)Thread.State.TERMINATED), (Object)((Object)this.thread.getState()));
        }

        void joinSuccessfully(long timeoutMillis) {
            Uninterruptibles.joinUninterruptibly((Thread)this.thread, (long)timeoutMillis, (TimeUnit)TimeUnit.MILLISECONDS);
            this.completed.assertCompletionExpected();
            Assert.assertEquals((Object)((Object)Thread.State.TERMINATED), (Object)((Object)this.thread.getState()));
        }

        void joinUnsuccessfully(long timeoutMillis) {
            Uninterruptibles.joinUninterruptibly((Thread)this.thread, (long)timeoutMillis, (TimeUnit)TimeUnit.MILLISECONDS);
            this.completed.assertCompletionNotExpected(timeoutMillis);
            Assert.assertFalse((boolean)Thread.State.TERMINATED.equals((Object)this.thread.getState()));
        }
    }

    private static class EnableReads
    extends DelayedActionRunnable {
        private final BlockingQueue<String> queue;

        public EnableReads(BlockingQueue<String> queue, long tMinus) {
            super(tMinus);
            Assert.assertTrue((boolean)queue.isEmpty());
            this.queue = queue;
        }

        @Override
        protected void doAction() {
            Assert.assertTrue((boolean)this.queue.offer(UninterruptiblesTest.EXPECTED_TAKE));
        }
    }

    private static class EnableWrites
    extends DelayedActionRunnable {
        private final BlockingQueue<String> queue;

        public EnableWrites(BlockingQueue<String> queue, long tMinus) {
            super(tMinus);
            Assert.assertFalse((boolean)queue.isEmpty());
            Assert.assertFalse((boolean)queue.offer("shouldBeRejected"));
            this.queue = queue;
        }

        @Override
        protected void doAction() {
            Assert.assertNotNull(this.queue.remove());
        }
    }

    private static class CountDown
    extends DelayedActionRunnable {
        private final CountDownLatch latch;

        public CountDown(CountDownLatch latch, long tMinus) {
            super(tMinus);
            this.latch = latch;
        }

        @Override
        protected void doAction() {
            this.latch.countDown();
        }
    }

    private static abstract class DelayedActionRunnable
    implements Runnable {
        private final long tMinus;

        protected DelayedActionRunnable(long tMinus) {
            this.tMinus = tMinus;
        }

        @Override
        public final void run() {
            try {
                Thread.sleep(this.tMinus);
            }
            catch (InterruptedException e) {
                throw new AssertionError((Object)e);
            }
            this.doAction();
        }

        protected abstract void doAction();
    }

    private static final class TimedSemaphore {
        final Semaphore semaphore = new Semaphore(0);
        final Completion completed;

        static TimedSemaphore createWithDelay(long countdownInMillis) {
            return new TimedSemaphore(countdownInMillis);
        }

        private TimedSemaphore(long countdownInMillis) {
            this.completed = new Completion(countdownInMillis);
            this.scheduleRelease(countdownInMillis);
        }

        void tryAcquireSuccessfully(long timeoutMillis) {
            Assert.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)this.semaphore, (long)timeoutMillis, (TimeUnit)TimeUnit.MILLISECONDS));
            this.completed.assertCompletionExpected();
        }

        private void tryAcquireUnsuccessfully(long timeoutMillis) {
            Assert.assertFalse((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)this.semaphore, (long)timeoutMillis, (TimeUnit)TimeUnit.MILLISECONDS));
            this.completed.assertCompletionNotExpected(timeoutMillis);
        }

        void tryAcquireSuccessfully(int permits, long timeoutMillis) {
            Assert.assertTrue((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)this.semaphore, (int)permits, (long)timeoutMillis, (TimeUnit)TimeUnit.MILLISECONDS));
            this.completed.assertCompletionExpected();
        }

        private void tryAcquireUnsuccessfully(int permits, long timeoutMillis) {
            Assert.assertFalse((boolean)Uninterruptibles.tryAcquireUninterruptibly((Semaphore)this.semaphore, (int)permits, (long)timeoutMillis, (TimeUnit)TimeUnit.MILLISECONDS));
            this.completed.assertCompletionNotExpected(timeoutMillis);
        }

        private void scheduleRelease(long countdownInMillis) {
            Release toRun = new Release(this.semaphore, countdownInMillis);
            Thread releaserThread = new Thread(toRun);
            releaserThread.start();
        }
    }

    private static final class TimedTakeQueue {
        final BlockingQueue<String> queue = new ArrayBlockingQueue<String>(1);
        final Completion completed;

        static TimedTakeQueue createWithDelay(long countdownInMillis) {
            return new TimedTakeQueue(countdownInMillis);
        }

        private TimedTakeQueue(long countdownInMillis) {
            this.completed = new Completion(countdownInMillis);
            TimedTakeQueue.scheduleEnableReads(this.queue, countdownInMillis);
        }

        void takeSuccessfully() {
            Assert.assertEquals((String)UninterruptiblesTest.EXPECTED_TAKE, (String)((String)Uninterruptibles.takeUninterruptibly(this.queue)));
            this.completed.assertCompletionExpected();
            Assert.assertTrue((boolean)this.queue.isEmpty());
        }

        private static void scheduleEnableReads(BlockingQueue<String> queue, long countdownInMillis) {
            EnableReads toRun = new EnableReads(queue, countdownInMillis);
            Thread enablerThread = new Thread(toRun);
            enablerThread.start();
        }
    }

    private static final class TimedPutQueue {
        final BlockingQueue<String> queue = new ArrayBlockingQueue<String>(1);
        final Completion completed;

        static TimedPutQueue createWithDelay(long countdownInMillis) {
            return new TimedPutQueue(countdownInMillis);
        }

        private TimedPutQueue(long countdownInMillis) {
            Assert.assertTrue((boolean)this.queue.offer("blocksPutCallsUntilRemoved"));
            this.completed = new Completion(countdownInMillis);
            TimedPutQueue.scheduleEnableWrites(this.queue, countdownInMillis);
        }

        void putSuccessfully() {
            Uninterruptibles.putUninterruptibly(this.queue, (Object)"");
            this.completed.assertCompletionExpected();
            Assert.assertEquals((String)"", (String)((String)this.queue.peek()));
        }

        private static void scheduleEnableWrites(BlockingQueue<String> queue, long countdownInMillis) {
            EnableWrites toRun = new EnableWrites(queue, countdownInMillis);
            Thread enablerThread = new Thread(toRun);
            enablerThread.start();
        }
    }

    private static final class Completion {
        final Stopwatch stopwatch;
        final long expectedCompletionWaitMillis;

        Completion(long expectedCompletionWaitMillis) {
            this.expectedCompletionWaitMillis = expectedCompletionWaitMillis;
            this.stopwatch = Stopwatch.createStarted();
        }

        void assertCompletionExpected() {
            Completion.assertAtLeastTimePassed(this.stopwatch, this.expectedCompletionWaitMillis);
            UninterruptiblesTest.assertTimeNotPassed(this.stopwatch, this.expectedCompletionWaitMillis + 2500L);
        }

        void assertCompletionNotExpected(long timeout) {
            Preconditions.checkArgument((timeout < this.expectedCompletionWaitMillis ? 1 : 0) != 0);
            Completion.assertAtLeastTimePassed(this.stopwatch, timeout);
            UninterruptiblesTest.assertTimeNotPassed(this.stopwatch, this.expectedCompletionWaitMillis);
        }

        private static void assertAtLeastTimePassed(Stopwatch stopwatch, long expectedMillis) {
            long elapsedMillis = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            Assert.assertTrue((String)("Expected elapsed millis to be >= " + expectedMillis + " but was " + elapsedMillis), (elapsedMillis + 5L >= expectedMillis ? 1 : 0) != 0);
        }
    }
}

