/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.io;

import com.google.common.base.Optional;
import com.google.common.collect.ObjectArrays;
import com.google.common.io.ByteSinkTester;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteSourceTester;
import com.google.common.io.CharSinkTester;
import com.google.common.io.CharSourceTester;
import com.google.common.io.InsecureRecursiveDeleteException;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import com.google.common.io.SourceSinkFactories;
import com.google.common.jimfs.Configuration;
import com.google.common.jimfs.Feature;
import com.google.common.jimfs.Jimfs;
import com.google.common.truth.Truth;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.EnumSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

public class MoreFilesTest
extends TestCase {
    private static final FileSystem FS = FileSystems.getDefault();
    private Path tempDir;

    public static TestSuite suite() {
        TestSuite suite = new TestSuite();
        suite.addTest((Test)ByteSourceTester.tests("MoreFiles.asByteSource[Path]", SourceSinkFactories.pathByteSourceFactory(), true));
        suite.addTest((Test)ByteSinkTester.tests("MoreFiles.asByteSink[Path]", SourceSinkFactories.pathByteSinkFactory()));
        suite.addTest((Test)ByteSinkTester.tests("MoreFiles.asByteSink[Path, APPEND]", SourceSinkFactories.appendingPathByteSinkFactory()));
        suite.addTest((Test)CharSourceTester.tests("MoreFiles.asCharSource[Path, Charset]", SourceSinkFactories.pathCharSourceFactory(), false));
        suite.addTest((Test)CharSinkTester.tests("MoreFiles.asCharSink[Path, Charset]", SourceSinkFactories.pathCharSinkFactory()));
        suite.addTest((Test)CharSinkTester.tests("MoreFiles.asCharSink[Path, Charset, APPEND]", SourceSinkFactories.appendingPathCharSinkFactory()));
        suite.addTestSuite(MoreFilesTest.class);
        return suite;
    }

    private static Path root() {
        return FS.getRootDirectories().iterator().next();
    }

    protected void setUp() throws Exception {
        this.tempDir = Files.createTempDirectory("MoreFilesTest", new FileAttribute[0]);
    }

    protected void tearDown() throws Exception {
        if (this.tempDir != null) {
            Files.walkFileTree(this.tempDir, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    Files.deleteIfExists(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (exc != null) {
                        return FileVisitResult.TERMINATE;
                    }
                    Files.deleteIfExists(dir);
                    return FileVisitResult.CONTINUE;
                }
            });
        }
    }

    private Path createTempFile() throws IOException {
        return Files.createTempFile(this.tempDir, "test", ".test", new FileAttribute[0]);
    }

    public void testByteSource_size_ofDirectory() throws IOException {
        try (FileSystem fs = Jimfs.newFileSystem((Configuration)Configuration.unix());){
            Path dir = fs.getPath("dir", new String[0]);
            Files.createDirectory(dir, new FileAttribute[0]);
            ByteSource source = MoreFiles.asByteSource((Path)dir, (OpenOption[])new OpenOption[0]);
            Truth.assertThat((Optional)source.sizeIfKnown()).isAbsent();
            try {
                source.size();
                MoreFilesTest.fail();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public void testByteSource_size_ofSymlinkToDirectory() throws IOException {
        try (FileSystem fs = Jimfs.newFileSystem((Configuration)Configuration.unix());){
            Path dir = fs.getPath("dir", new String[0]);
            Files.createDirectory(dir, new FileAttribute[0]);
            Path link = fs.getPath("link", new String[0]);
            Files.createSymbolicLink(link, dir, new FileAttribute[0]);
            ByteSource source = MoreFiles.asByteSource((Path)link, (OpenOption[])new OpenOption[0]);
            Truth.assertThat((Optional)source.sizeIfKnown()).isAbsent();
            try {
                source.size();
                MoreFilesTest.fail();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public void testByteSource_size_ofSymlinkToRegularFile() throws IOException {
        try (FileSystem fs = Jimfs.newFileSystem((Configuration)Configuration.unix());){
            Path file = fs.getPath("file", new String[0]);
            Files.write(file, new byte[10], new OpenOption[0]);
            Path link = fs.getPath("link", new String[0]);
            Files.createSymbolicLink(link, file, new FileAttribute[0]);
            ByteSource source = MoreFiles.asByteSource((Path)link, (OpenOption[])new OpenOption[0]);
            MoreFilesTest.assertEquals((long)10L, (long)((Long)source.sizeIfKnown().get()));
            MoreFilesTest.assertEquals((long)10L, (long)source.size());
        }
    }

    public void testByteSource_size_ofSymlinkToRegularFile_nofollowLinks() throws IOException {
        try (FileSystem fs = Jimfs.newFileSystem((Configuration)Configuration.unix());){
            Path file = fs.getPath("file", new String[0]);
            Files.write(file, new byte[10], new OpenOption[0]);
            Path link = fs.getPath("link", new String[0]);
            Files.createSymbolicLink(link, file, new FileAttribute[0]);
            ByteSource source = MoreFiles.asByteSource((Path)link, (OpenOption[])new OpenOption[]{LinkOption.NOFOLLOW_LINKS});
            Truth.assertThat((Optional)source.sizeIfKnown()).isAbsent();
            try {
                source.size();
                MoreFilesTest.fail();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public void testEqual() throws IOException {
        try (FileSystem fs = Jimfs.newFileSystem((Configuration)Configuration.unix());){
            Path fooPath = fs.getPath("foo", new String[0]);
            Path barPath = fs.getPath("bar", new String[0]);
            MoreFiles.asCharSink((Path)fooPath, (Charset)StandardCharsets.UTF_8, (OpenOption[])new OpenOption[0]).write((CharSequence)"foo");
            MoreFiles.asCharSink((Path)barPath, (Charset)StandardCharsets.UTF_8, (OpenOption[])new OpenOption[0]).write((CharSequence)"barbar");
            Truth.assertThat((Boolean)MoreFiles.equal((Path)fooPath, (Path)barPath)).isFalse();
            Truth.assertThat((Boolean)MoreFiles.equal((Path)fooPath, (Path)fooPath)).isTrue();
            Truth.assertThat((Boolean)MoreFiles.asByteSource((Path)fooPath, (OpenOption[])new OpenOption[0]).contentEquals(MoreFiles.asByteSource((Path)fooPath, (OpenOption[])new OpenOption[0]))).isTrue();
            Path fooCopy = Files.copy(fooPath, fs.getPath("fooCopy", new String[0]), new CopyOption[0]);
            Truth.assertThat((Boolean)Files.isSameFile(fooPath, fooCopy)).isFalse();
            Truth.assertThat((Boolean)MoreFiles.equal((Path)fooPath, (Path)fooCopy)).isTrue();
            MoreFiles.asCharSink((Path)fooCopy, (Charset)StandardCharsets.UTF_8, (OpenOption[])new OpenOption[0]).write((CharSequence)"boo");
            Truth.assertThat((Long)MoreFiles.asByteSource((Path)fooPath, (OpenOption[])new OpenOption[0]).size()).isEqualTo((Object)MoreFiles.asByteSource((Path)fooCopy, (OpenOption[])new OpenOption[0]).size());
            Truth.assertThat((Boolean)MoreFiles.equal((Path)fooPath, (Path)fooCopy)).isFalse();
        }
    }

    public void testEqual_links() throws IOException {
        try (FileSystem fs = Jimfs.newFileSystem((Configuration)Configuration.unix());){
            Path fooPath = fs.getPath("foo", new String[0]);
            MoreFiles.asCharSink((Path)fooPath, (Charset)StandardCharsets.UTF_8, (OpenOption[])new OpenOption[0]).write((CharSequence)"foo");
            Path fooSymlink = fs.getPath("symlink", new String[0]);
            Files.createSymbolicLink(fooSymlink, fooPath, new FileAttribute[0]);
            Path fooHardlink = fs.getPath("hardlink", new String[0]);
            Files.createLink(fooHardlink, fooPath);
            Truth.assertThat((Boolean)MoreFiles.equal((Path)fooPath, (Path)fooSymlink)).isTrue();
            Truth.assertThat((Boolean)MoreFiles.equal((Path)fooPath, (Path)fooHardlink)).isTrue();
            Truth.assertThat((Boolean)MoreFiles.equal((Path)fooSymlink, (Path)fooHardlink)).isTrue();
        }
    }

    public void testTouch() throws IOException {
        Path temp = this.createTempFile();
        MoreFilesTest.assertTrue((boolean)Files.exists(temp, new LinkOption[0]));
        Files.delete(temp);
        MoreFilesTest.assertFalse((boolean)Files.exists(temp, new LinkOption[0]));
        MoreFiles.touch((Path)temp);
        MoreFilesTest.assertTrue((boolean)Files.exists(temp, new LinkOption[0]));
        MoreFiles.touch((Path)temp);
        MoreFilesTest.assertTrue((boolean)Files.exists(temp, new LinkOption[0]));
    }

    public void testTouchTime() throws IOException {
        Path temp = this.createTempFile();
        MoreFilesTest.assertTrue((boolean)Files.exists(temp, new LinkOption[0]));
        Files.setLastModifiedTime(temp, FileTime.fromMillis(0L));
        MoreFilesTest.assertEquals((long)0L, (long)Files.getLastModifiedTime(temp, new LinkOption[0]).toMillis());
        MoreFiles.touch((Path)temp);
        Truth.assertThat((Long)Files.getLastModifiedTime(temp, new LinkOption[0]).toMillis()).isNotEqualTo((Object)0);
    }

    public void testCreateParentDirectories_root() throws IOException {
        Path root = MoreFilesTest.root();
        MoreFilesTest.assertNull((Object)root.getParent());
        MoreFilesTest.assertNull((Object)root.toRealPath(new LinkOption[0]).getParent());
        MoreFiles.createParentDirectories((Path)root, (FileAttribute[])new FileAttribute[0]);
    }

    public void testCreateParentDirectories_relativePath() throws IOException {
        Path path = FS.getPath("nonexistent.file", new String[0]);
        MoreFilesTest.assertNull((Object)path.getParent());
        MoreFilesTest.assertNotNull((Object)path.toAbsolutePath().getParent());
        MoreFiles.createParentDirectories((Path)path, (FileAttribute[])new FileAttribute[0]);
    }

    public void testCreateParentDirectories_noParentsNeeded() throws IOException {
        Path path = this.tempDir.resolve("nonexistent.file");
        MoreFilesTest.assertTrue((boolean)Files.exists(path.getParent(), new LinkOption[0]));
        MoreFiles.createParentDirectories((Path)path, (FileAttribute[])new FileAttribute[0]);
    }

    public void testCreateParentDirectories_oneParentNeeded() throws IOException {
        Path path = this.tempDir.resolve("parent/nonexistent.file");
        Path parent = path.getParent();
        MoreFilesTest.assertFalse((boolean)Files.exists(parent, new LinkOption[0]));
        MoreFiles.createParentDirectories((Path)path, (FileAttribute[])new FileAttribute[0]);
        MoreFilesTest.assertTrue((boolean)Files.exists(parent, new LinkOption[0]));
    }

    public void testCreateParentDirectories_multipleParentsNeeded() throws IOException {
        Path path = this.tempDir.resolve("grandparent/parent/nonexistent.file");
        Path parent = path.getParent();
        Path grandparent = parent.getParent();
        MoreFilesTest.assertFalse((boolean)Files.exists(grandparent, new LinkOption[0]));
        MoreFilesTest.assertFalse((boolean)Files.exists(parent, new LinkOption[0]));
        MoreFiles.createParentDirectories((Path)path, (FileAttribute[])new FileAttribute[0]);
        MoreFilesTest.assertTrue((boolean)Files.exists(parent, new LinkOption[0]));
        MoreFilesTest.assertTrue((boolean)Files.exists(grandparent, new LinkOption[0]));
    }

    public void testCreateParentDirectories_noPermission() {
        Path file = MoreFilesTest.root().resolve("parent/nonexistent.file");
        Path parent = file.getParent();
        MoreFilesTest.assertFalse((boolean)Files.exists(parent, new LinkOption[0]));
        try {
            MoreFiles.createParentDirectories((Path)file, (FileAttribute[])new FileAttribute[0]);
            Files.delete(parent);
            MoreFilesTest.fail((String)"expected exception");
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public void testCreateParentDirectories_nonDirectoryParentExists() throws IOException {
        Path parent = this.createTempFile();
        MoreFilesTest.assertTrue((boolean)Files.isRegularFile(parent, new LinkOption[0]));
        Path file = parent.resolve("foo");
        try {
            MoreFiles.createParentDirectories((Path)file, (FileAttribute[])new FileAttribute[0]);
            MoreFilesTest.fail();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public void testCreateParentDirectories_symlinkParentExists() throws IOException {
        Path symlink = this.tempDir.resolve("linkToDir");
        Files.createSymbolicLink(symlink, MoreFilesTest.root(), new FileAttribute[0]);
        Path file = symlink.resolve("foo");
        MoreFiles.createParentDirectories((Path)file, (FileAttribute[])new FileAttribute[0]);
    }

    public void testGetFileExtension() {
        MoreFilesTest.assertEquals((String)"txt", (String)MoreFiles.getFileExtension((Path)FS.getPath(".txt", new String[0])));
        MoreFilesTest.assertEquals((String)"txt", (String)MoreFiles.getFileExtension((Path)FS.getPath("blah.txt", new String[0])));
        MoreFilesTest.assertEquals((String)"txt", (String)MoreFiles.getFileExtension((Path)FS.getPath("blah..txt", new String[0])));
        MoreFilesTest.assertEquals((String)"txt", (String)MoreFiles.getFileExtension((Path)FS.getPath(".blah.txt", new String[0])));
        MoreFilesTest.assertEquals((String)"txt", (String)MoreFiles.getFileExtension((Path)MoreFilesTest.root().resolve("tmp/blah.txt")));
        MoreFilesTest.assertEquals((String)"gz", (String)MoreFiles.getFileExtension((Path)FS.getPath("blah.tar.gz", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)MoreFilesTest.root()));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)FS.getPath(".", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)FS.getPath("..", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)FS.getPath("...", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)FS.getPath("blah", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)FS.getPath("blah.", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)FS.getPath(".blah.", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)MoreFilesTest.root().resolve("foo.bar/blah")));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getFileExtension((Path)MoreFilesTest.root().resolve("foo/.bar/blah")));
    }

    public void testGetNameWithoutExtension() {
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath(".txt", new String[0])));
        MoreFilesTest.assertEquals((String)"blah", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath("blah.txt", new String[0])));
        MoreFilesTest.assertEquals((String)"blah.", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath("blah..txt", new String[0])));
        MoreFilesTest.assertEquals((String)".blah", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath(".blah.txt", new String[0])));
        MoreFilesTest.assertEquals((String)"blah", (String)MoreFiles.getNameWithoutExtension((Path)MoreFilesTest.root().resolve("tmp/blah.txt")));
        MoreFilesTest.assertEquals((String)"blah.tar", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath("blah.tar.gz", new String[0])));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getNameWithoutExtension((Path)MoreFilesTest.root()));
        MoreFilesTest.assertEquals((String)"", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath(".", new String[0])));
        MoreFilesTest.assertEquals((String)".", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath("..", new String[0])));
        MoreFilesTest.assertEquals((String)"..", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath("...", new String[0])));
        MoreFilesTest.assertEquals((String)"blah", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath("blah", new String[0])));
        MoreFilesTest.assertEquals((String)"blah", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath("blah.", new String[0])));
        MoreFilesTest.assertEquals((String)".blah", (String)MoreFiles.getNameWithoutExtension((Path)FS.getPath(".blah.", new String[0])));
        MoreFilesTest.assertEquals((String)"blah", (String)MoreFiles.getNameWithoutExtension((Path)MoreFilesTest.root().resolve("foo.bar/blah")));
        MoreFilesTest.assertEquals((String)"blah", (String)MoreFiles.getNameWithoutExtension((Path)MoreFilesTest.root().resolve("foo/.bar/blah")));
    }

    public void testPredicates() throws IOException {
        Path file = this.createTempFile();
        Path dir = this.tempDir.resolve("dir");
        Files.createDirectory(dir, new FileAttribute[0]);
        MoreFilesTest.assertTrue((boolean)MoreFiles.isDirectory((LinkOption[])new LinkOption[0]).apply((Object)dir));
        MoreFilesTest.assertFalse((boolean)MoreFiles.isRegularFile((LinkOption[])new LinkOption[0]).apply((Object)dir));
        MoreFilesTest.assertFalse((boolean)MoreFiles.isDirectory((LinkOption[])new LinkOption[0]).apply((Object)file));
        MoreFilesTest.assertTrue((boolean)MoreFiles.isRegularFile((LinkOption[])new LinkOption[0]).apply((Object)file));
        Path symlinkToDir = this.tempDir.resolve("symlinkToDir");
        Path symlinkToFile = this.tempDir.resolve("symlinkToFile");
        Files.createSymbolicLink(symlinkToDir, dir, new FileAttribute[0]);
        Files.createSymbolicLink(symlinkToFile, file, new FileAttribute[0]);
        MoreFilesTest.assertTrue((boolean)MoreFiles.isDirectory((LinkOption[])new LinkOption[0]).apply((Object)symlinkToDir));
        MoreFilesTest.assertFalse((boolean)MoreFiles.isRegularFile((LinkOption[])new LinkOption[0]).apply((Object)symlinkToDir));
        MoreFilesTest.assertFalse((boolean)MoreFiles.isDirectory((LinkOption[])new LinkOption[0]).apply((Object)symlinkToFile));
        MoreFilesTest.assertTrue((boolean)MoreFiles.isRegularFile((LinkOption[])new LinkOption[0]).apply((Object)symlinkToFile));
        MoreFilesTest.assertFalse((boolean)MoreFiles.isDirectory((LinkOption[])new LinkOption[]{LinkOption.NOFOLLOW_LINKS}).apply((Object)symlinkToDir));
        MoreFilesTest.assertFalse((boolean)MoreFiles.isRegularFile((LinkOption[])new LinkOption[]{LinkOption.NOFOLLOW_LINKS}).apply((Object)symlinkToFile));
    }

    static FileSystem newTestFileSystem(Feature ... supportedFeatures) throws IOException {
        FileSystem fs = Jimfs.newFileSystem((Configuration)Configuration.unix().toBuilder().setSupportedFeatures((Feature[])ObjectArrays.concat((Object)Feature.SYMBOLIC_LINKS, (Object[])supportedFeatures)).build());
        Files.createDirectories(fs.getPath("dir/b/i/j/l", new String[0]), new FileAttribute[0]);
        Files.createFile(fs.getPath("dir/a", new String[0]), new FileAttribute[0]);
        Files.createFile(fs.getPath("dir/c", new String[0]), new FileAttribute[0]);
        Files.createSymbolicLink(fs.getPath("dir/d", new String[0]), fs.getPath("b/i", new String[0]), new FileAttribute[0]);
        Files.createDirectory(fs.getPath("dir/e", new String[0]), new FileAttribute[0]);
        Files.createSymbolicLink(fs.getPath("dir/f", new String[0]), fs.getPath("/dontdelete", new String[0]), new FileAttribute[0]);
        Files.createFile(fs.getPath("dir/b/g", new String[0]), new FileAttribute[0]);
        Files.createSymbolicLink(fs.getPath("dir/b/h", new String[0]), fs.getPath("../a", new String[0]), new FileAttribute[0]);
        Files.createFile(fs.getPath("dir/b/i/j/k", new String[0]), new FileAttribute[0]);
        Files.createDirectory(fs.getPath("/dontdelete", new String[0]), new FileAttribute[0]);
        Files.createFile(fs.getPath("/dontdelete/a", new String[0]), new FileAttribute[0]);
        Files.createDirectory(fs.getPath("/dontdelete/b", new String[0]), new FileAttribute[0]);
        Files.createFile(fs.getPath("/dontdelete/c", new String[0]), new FileAttribute[0]);
        Files.createSymbolicLink(fs.getPath("/symlinktodir", new String[0]), fs.getPath("work/dir", new String[0]), new FileAttribute[0]);
        return fs;
    }

    public void testDirectoryDeletion_basic() throws IOException {
        for (DirectoryDeleteMethod method : EnumSet.allOf(DirectoryDeleteMethod.class)) {
            FileSystem fs = MoreFilesTest.newTestFileSystem(Feature.SECURE_DIRECTORY_STREAM);
            Throwable throwable = null;
            try {
                Path dir = fs.getPath("dir", new String[0]);
                MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
                method.delete(dir, new RecursiveDeleteOption[0]);
                method.assertDeleteSucceeded(dir);
                MoreFilesTest.assertEquals((String)("contents of /dontdelete deleted by delete method " + (Object)((Object)method)), (int)3, (int)MoreFiles.listFiles((Path)fs.getPath("/dontdelete", new String[0])).size());
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (fs == null) continue;
                if (throwable != null) {
                    try {
                        fs.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                fs.close();
            }
        }
    }

    public void testDirectoryDeletion_emptyDir() throws IOException {
        for (DirectoryDeleteMethod method : EnumSet.allOf(DirectoryDeleteMethod.class)) {
            FileSystem fs = MoreFilesTest.newTestFileSystem(Feature.SECURE_DIRECTORY_STREAM);
            Throwable throwable = null;
            try {
                Path emptyDir = fs.getPath("dir/e", new String[0]);
                MoreFilesTest.assertEquals((int)0, (int)MoreFiles.listFiles((Path)emptyDir).size());
                method.delete(emptyDir, new RecursiveDeleteOption[0]);
                method.assertDeleteSucceeded(emptyDir);
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (fs == null) continue;
                if (throwable != null) {
                    try {
                        fs.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                fs.close();
            }
        }
    }

    public void testDeleteRecursively_symlinkToDir() throws IOException {
        try (FileSystem fs = MoreFilesTest.newTestFileSystem(Feature.SECURE_DIRECTORY_STREAM);){
            Path symlink = fs.getPath("/symlinktodir", new String[0]);
            Path dir = fs.getPath("dir", new String[0]);
            MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
            MoreFiles.deleteRecursively((Path)symlink, (RecursiveDeleteOption[])new RecursiveDeleteOption[0]);
            MoreFilesTest.assertFalse((boolean)Files.exists(symlink, new LinkOption[0]));
            MoreFilesTest.assertTrue((boolean)Files.exists(dir, new LinkOption[0]));
            MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
        }
    }

    public void testDeleteDirectoryContents_symlinkToDir() throws IOException {
        try (FileSystem fs = MoreFilesTest.newTestFileSystem(Feature.SECURE_DIRECTORY_STREAM);){
            Path symlink = fs.getPath("/symlinktodir", new String[0]);
            Path dir = fs.getPath("dir", new String[0]);
            MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)symlink).size());
            MoreFiles.deleteDirectoryContents((Path)symlink, (RecursiveDeleteOption[])new RecursiveDeleteOption[0]);
            MoreFilesTest.assertTrue((boolean)Files.exists(symlink, LinkOption.NOFOLLOW_LINKS));
            MoreFilesTest.assertTrue((boolean)Files.exists(symlink, new LinkOption[0]));
            MoreFilesTest.assertTrue((boolean)Files.exists(dir, new LinkOption[0]));
            MoreFilesTest.assertEquals((int)0, (int)MoreFiles.listFiles((Path)symlink).size());
        }
    }

    public void testDirectoryDeletion_sdsNotSupported_fails() throws IOException {
        for (DirectoryDeleteMethod method : EnumSet.allOf(DirectoryDeleteMethod.class)) {
            FileSystem fs = MoreFilesTest.newTestFileSystem(new Feature[0]);
            Throwable throwable = null;
            try {
                Path dir = fs.getPath("dir", new String[0]);
                MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
                try {
                    method.delete(dir, new RecursiveDeleteOption[0]);
                    MoreFilesTest.fail((String)"expected InsecureRecursiveDeleteException");
                }
                catch (InsecureRecursiveDeleteException insecureRecursiveDeleteException) {
                    // empty catch block
                }
                MoreFilesTest.assertTrue((boolean)Files.exists(dir, new LinkOption[0]));
                MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (fs == null) continue;
                if (throwable != null) {
                    try {
                        fs.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                fs.close();
            }
        }
    }

    public void testDirectoryDeletion_sdsNotSupported_allowInsecure() throws IOException {
        for (DirectoryDeleteMethod method : EnumSet.allOf(DirectoryDeleteMethod.class)) {
            FileSystem fs = MoreFilesTest.newTestFileSystem(new Feature[0]);
            Throwable throwable = null;
            try {
                Path dir = fs.getPath("dir", new String[0]);
                MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
                method.delete(dir, RecursiveDeleteOption.ALLOW_INSECURE);
                method.assertDeleteSucceeded(dir);
                MoreFilesTest.assertEquals((String)("contents of /dontdelete deleted by delete method " + (Object)((Object)method)), (int)3, (int)MoreFiles.listFiles((Path)fs.getPath("/dontdelete", new String[0])).size());
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (fs == null) continue;
                if (throwable != null) {
                    try {
                        fs.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                fs.close();
            }
        }
    }

    public void testDeleteRecursively_symlinkToDir_sdsNotSupported_allowInsecure() throws IOException {
        try (FileSystem fs = MoreFilesTest.newTestFileSystem(new Feature[0]);){
            Path symlink = fs.getPath("/symlinktodir", new String[0]);
            Path dir = fs.getPath("dir", new String[0]);
            MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
            MoreFiles.deleteRecursively((Path)symlink, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            MoreFilesTest.assertFalse((boolean)Files.exists(symlink, new LinkOption[0]));
            MoreFilesTest.assertTrue((boolean)Files.exists(dir, new LinkOption[0]));
            MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
        }
    }

    public void testDeleteDirectoryContents_symlinkToDir_sdsNotSupported_allowInsecure() throws IOException {
        try (FileSystem fs = MoreFilesTest.newTestFileSystem(new Feature[0]);){
            Path symlink = fs.getPath("/symlinktodir", new String[0]);
            Path dir = fs.getPath("dir", new String[0]);
            MoreFilesTest.assertEquals((int)6, (int)MoreFiles.listFiles((Path)dir).size());
            MoreFiles.deleteDirectoryContents((Path)symlink, (RecursiveDeleteOption[])new RecursiveDeleteOption[]{RecursiveDeleteOption.ALLOW_INSECURE});
            MoreFilesTest.assertEquals((int)0, (int)MoreFiles.listFiles((Path)dir).size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testDirectoryDeletion_directorySymlinkRace() throws IOException {
        for (DirectoryDeleteMethod method : EnumSet.allOf(DirectoryDeleteMethod.class)) {
            FileSystem fs = MoreFilesTest.newTestFileSystem(Feature.SECURE_DIRECTORY_STREAM);
            Throwable throwable = null;
            try {
                Path dirToDelete = fs.getPath("dir/b/i", new String[0]);
                Path changingFile = dirToDelete.resolve("j/l");
                Path symlinkTarget = fs.getPath("/dontdelete", new String[0]);
                ExecutorService executor = Executors.newSingleThreadExecutor();
                MoreFilesTest.startDirectorySymlinkSwitching(changingFile, symlinkTarget, executor);
                try {
                    for (int i = 0; i < 5000; ++i) {
                        try {
                            Files.createDirectories(changingFile, new FileAttribute[0]);
                            Files.createFile(dirToDelete.resolve("j/k"), new FileAttribute[0]);
                        }
                        catch (FileAlreadyExistsException fileAlreadyExistsException) {
                            // empty catch block
                        }
                        try {
                            method.delete(dirToDelete, new RecursiveDeleteOption[0]);
                        }
                        catch (FileSystemException fileSystemException) {
                            // empty catch block
                        }
                        MoreFilesTest.assertEquals((int)3, (int)MoreFiles.listFiles((Path)symlinkTarget).size());
                        Thread.yield();
                    }
                }
                finally {
                    executor.shutdownNow();
                }
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (fs == null) continue;
                if (throwable != null) {
                    try {
                        fs.close();
                    }
                    catch (Throwable throwable3) {
                        throwable.addSuppressed(throwable3);
                    }
                    continue;
                }
                fs.close();
            }
        }
    }

    public void testDeleteRecursively_nonDirectoryFile() throws IOException {
        try (FileSystem fs = MoreFilesTest.newTestFileSystem(Feature.SECURE_DIRECTORY_STREAM);){
            Path file = fs.getPath("dir/a", new String[0]);
            MoreFilesTest.assertTrue((boolean)Files.isRegularFile(file, LinkOption.NOFOLLOW_LINKS));
            MoreFiles.deleteRecursively((Path)file, (RecursiveDeleteOption[])new RecursiveDeleteOption[0]);
            MoreFilesTest.assertFalse((boolean)Files.exists(file, LinkOption.NOFOLLOW_LINKS));
            Path symlink = fs.getPath("/symlinktodir", new String[0]);
            MoreFilesTest.assertTrue((boolean)Files.isSymbolicLink(symlink));
            Path realSymlinkTarget = symlink.toRealPath(new LinkOption[0]);
            MoreFilesTest.assertTrue((boolean)Files.isDirectory(realSymlinkTarget, LinkOption.NOFOLLOW_LINKS));
            MoreFiles.deleteRecursively((Path)symlink, (RecursiveDeleteOption[])new RecursiveDeleteOption[0]);
            MoreFilesTest.assertFalse((boolean)Files.exists(symlink, LinkOption.NOFOLLOW_LINKS));
            MoreFilesTest.assertTrue((boolean)Files.isDirectory(realSymlinkTarget, LinkOption.NOFOLLOW_LINKS));
        }
    }

    private static void startDirectorySymlinkSwitching(final Path file, final Path target, ExecutorService executor) {
        Future<?> possiblyIgnoredError = executor.submit(new Runnable(){

            @Override
            public void run() {
                boolean createSymlink = false;
                while (!Thread.interrupted()) {
                    try {
                        if (Files.deleteIfExists(file)) {
                            if (createSymlink) {
                                Files.createSymbolicLink(file, target, new FileAttribute[0]);
                            } else {
                                Files.createDirectory(file, new FileAttribute[0]);
                            }
                            createSymlink = !createSymlink;
                        }
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                    Thread.yield();
                }
            }
        });
    }

    private static enum DirectoryDeleteMethod {
        DELETE_DIRECTORY_CONTENTS{

            @Override
            public void delete(Path path, RecursiveDeleteOption ... options) throws IOException {
                MoreFiles.deleteDirectoryContents((Path)path, (RecursiveDeleteOption[])options);
            }

            @Override
            public void assertDeleteSucceeded(Path path) throws IOException {
                TestCase.assertEquals((String)("contents of directory " + path + " not deleted with delete method " + (Object)((Object)this)), (int)0, (int)MoreFiles.listFiles((Path)path).size());
            }
        }
        ,
        DELETE_RECURSIVELY{

            @Override
            public void delete(Path path, RecursiveDeleteOption ... options) throws IOException {
                MoreFiles.deleteRecursively((Path)path, (RecursiveDeleteOption[])options);
            }

            @Override
            public void assertDeleteSucceeded(Path path) throws IOException {
                TestCase.assertFalse((String)("file " + path + " not deleted with delete method " + (Object)((Object)this)), (boolean)Files.exists(path, new LinkOption[0]));
            }
        };


        public abstract void delete(Path var1, RecursiveDeleteOption ... var2) throws IOException;

        public abstract void assertDeleteSucceeded(Path var1) throws IOException;
    }
}

