/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.reflect;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.StandardSystemProperty;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.io.Closer;
import com.google.common.io.MoreFiles;
import com.google.common.io.RecursiveDeleteOption;
import com.google.common.io.Resources;
import com.google.common.reflect.AndroidIncompatible;
import com.google.common.reflect.ClassPath;
import com.google.common.testing.EqualsTester;
import com.google.common.testing.NullPointerTester;
import com.google.common.truth.Truth;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FilePermission;
import java.io.IOException;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.Permission;
import java.security.PermissionCollection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import junit.framework.TestCase;
import org.junit.Test;

public class ClassPathTest
extends TestCase {
    private static final Logger log = Logger.getLogger(ClassPathTest.class.getName());

    public void testEquals() {
        new EqualsTester().addEqualityGroup(new Object[]{ClassPathTest.classInfo(ClassPathTest.class), ClassPathTest.classInfo(ClassPathTest.class)}).addEqualityGroup(new Object[]{ClassPathTest.classInfo(Test.class), ClassPathTest.classInfo(Test.class, ((Object)((Object)this)).getClass().getClassLoader())}).addEqualityGroup(new Object[]{new ClassPath.ResourceInfo("a/b/c.txt", ((Object)((Object)this)).getClass().getClassLoader()), new ClassPath.ResourceInfo("a/b/c.txt", ((Object)((Object)this)).getClass().getClassLoader())}).addEqualityGroup(new Object[]{new ClassPath.ResourceInfo("x.txt", ((Object)((Object)this)).getClass().getClassLoader())}).testEquals();
    }

    @AndroidIncompatible
    public void testClassPathEntries_emptyURLClassLoader_noParent() {
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathEntries((ClassLoader)new URLClassLoader(new URL[0], null)).keySet()).isEmpty();
    }

    @AndroidIncompatible
    public void testClassPathEntries_URLClassLoader_noParent() throws Exception {
        URL url1 = new URL("file:/a");
        URL url2 = new URL("file:/b");
        URLClassLoader classloader = new URLClassLoader(new URL[]{url1, url2}, null);
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)classloader)).containsExactly((Object)new File("/a"), (Object)classloader, new Object[]{new File("/b"), classloader});
    }

    @AndroidIncompatible
    public void testClassPathEntries_URLClassLoader_withParent() throws Exception {
        URL url1 = new URL("file:/a");
        URL url2 = new URL("file:/b");
        URLClassLoader parent = new URLClassLoader(new URL[]{url1}, null);
        URLClassLoader child = new URLClassLoader(new URL[]{url2}, (ClassLoader)parent){};
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)child)).containsExactly((Object)new File("/a"), (Object)parent, new Object[]{new File("/b"), child}).inOrder();
    }

    @AndroidIncompatible
    public void testClassPathEntries_duplicateUri_parentWins() throws Exception {
        URL url = new URL("file:/a");
        URLClassLoader parent = new URLClassLoader(new URL[]{url}, null);
        URLClassLoader child = new URLClassLoader(new URL[]{url}, (ClassLoader)parent){};
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)child)).containsExactly((Object)new File("/a"), (Object)parent, new Object[0]);
    }

    @AndroidIncompatible
    public void testClassPathEntries_notURLClassLoader_noParent() {
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)new ClassLoader(null){})).isEmpty();
    }

    @AndroidIncompatible
    public void testClassPathEntries_notURLClassLoader_withParent() throws Exception {
        URL url = new URL("file:/a");
        URLClassLoader parent = new URLClassLoader(new URL[]{url}, null);
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)new ClassLoader((ClassLoader)parent){})).containsExactly((Object)new File("/a"), (Object)parent, new Object[0]);
    }

    @AndroidIncompatible
    public void testClassPathEntries_notURLClassLoader_withParentAndGrandParent() throws Exception {
        URL url1 = new URL("file:/a");
        URL url2 = new URL("file:/b");
        URLClassLoader grandParent = new URLClassLoader(new URL[]{url1}, null);
        URLClassLoader parent = new URLClassLoader(new URL[]{url2}, (ClassLoader)grandParent);
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)new ClassLoader((ClassLoader)parent){})).containsExactly((Object)new File("/a"), (Object)grandParent, new Object[]{new File("/b"), parent});
    }

    @AndroidIncompatible
    public void testClassPathEntries_notURLClassLoader_withGrandParent() throws Exception {
        URL url = new URL("file:/a");
        URLClassLoader grandParent = new URLClassLoader(new URL[]{url}, null);
        ClassLoader parent = new ClassLoader((ClassLoader)grandParent){};
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)new ClassLoader(parent){})).containsExactly((Object)new File("/a"), (Object)grandParent, new Object[0]);
    }

    @AndroidIncompatible
    public void testClassPathEntries_URLClassLoader_pathWithSpace() throws Exception {
        URL url = new URL("file:///c:/Documents and Settings/");
        URLClassLoader classloader = new URLClassLoader(new URL[]{url}, null);
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)classloader)).containsExactly((Object)new File("/c:/Documents and Settings/"), (Object)classloader, new Object[0]);
    }

    @AndroidIncompatible
    public void testClassPathEntries_URLClassLoader_pathWithEscapedSpace() throws Exception {
        URL url = new URL("file:///c:/Documents%20and%20Settings/");
        URLClassLoader classloader = new URLClassLoader(new URL[]{url}, null);
        Truth.assertThat((Map)ClassPath.Scanner.getClassPathEntries((ClassLoader)classloader)).containsExactly((Object)new File("/c:/Documents and Settings/"), (Object)classloader, new Object[0]);
    }

    public void testToFile() throws Exception {
        Truth.assertThat((Comparable)ClassPath.toFile((URL)new URL("file:///c:/Documents%20and%20Settings/"))).isEqualTo((Object)new File("/c:/Documents and Settings/"));
        Truth.assertThat((Comparable)ClassPath.toFile((URL)new URL("file:///c:/Documents ~ Settings, or not/11-12 12:05"))).isEqualTo((Object)new File("/c:/Documents ~ Settings, or not/11-12 12:05"));
    }

    @AndroidIncompatible
    public void testToFile_AndroidIncompatible() throws Exception {
        Truth.assertThat((Comparable)ClassPath.toFile((URL)new URL("file:///c:\\Documents ~ Settings, or not\\11-12 12:05"))).isEqualTo((Object)new File("/c:\\Documents ~ Settings, or not\\11-12 12:05"));
        Truth.assertThat((Comparable)ClassPath.toFile((URL)new URL("file:///C:\\Program Files\\Apache Software Foundation"))).isEqualTo((Object)new File("/C:\\Program Files\\Apache Software Foundation/"));
        Truth.assertThat((Comparable)ClassPath.toFile((URL)new URL("file:///C:\\\u20320 \u22909"))).isEqualTo((Object)new File("/C:\\\u20320 \u22909"));
    }

    @AndroidIncompatible
    public void testJarFileWithSpaces() throws Exception {
        URL url = ClassPathTest.makeJarUrlWithName("To test unescaped spaces in jar file name.jar");
        URLClassLoader classloader = new URLClassLoader(new URL[]{url}, null);
        Truth.assertThat((Iterable)ClassPath.from((ClassLoader)classloader).getTopLevelClasses()).isNotEmpty();
    }

    public void testScan_classPathCycle() throws IOException {
        File jarFile = File.createTempFile("with_circular_class_path", ".jar");
        try {
            ClassPathTest.writeSelfReferencingJarFile(jarFile, "test.txt");
            ClassPath.DefaultScanner scanner = new ClassPath.DefaultScanner();
            scanner.scan(jarFile, ClassPathTest.class.getClassLoader());
            Truth.assertThat((Iterable)scanner.getResources()).hasSize(1);
        }
        finally {
            jarFile.delete();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @AndroidIncompatible
    public void testScanDirectory_symlinkCycle() throws IOException {
        ClassLoader loader = ClassPathTest.class.getClassLoader();
        Path root = Files.createTempDirectory("ClassPathTest", new FileAttribute[0]);
        try {
            Path left = Files.createDirectory(root.resolve("left"), new FileAttribute[0]);
            Files.createFile(left.resolve("some.txt"), new FileAttribute[0]);
            Path right = Files.createDirectory(root.resolve("right"), new FileAttribute[0]);
            Files.createFile(right.resolve("another.txt"), new FileAttribute[0]);
            Files.createSymbolicLink(left.resolve("sibling"), right, new FileAttribute[0]);
            Files.createSymbolicLink(right.resolve("sibling"), left, new FileAttribute[0]);
            ClassPath.DefaultScanner scanner = new ClassPath.DefaultScanner();
            scanner.scan(root.toFile(), loader);
            ClassPathTest.assertEquals((Object)ImmutableSet.of((Object)new ClassPath.ResourceInfo("left/some.txt", loader), (Object)new ClassPath.ResourceInfo("left/sibling/another.txt", loader), (Object)new ClassPath.ResourceInfo("right/another.txt", loader), (Object)new ClassPath.ResourceInfo("right/sibling/some.txt", loader)), (Object)scanner.getResources());
        }
        finally {
            ClassPathTest.deleteRecursivelyOrLog(root);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @AndroidIncompatible
    public void testScanDirectory_symlinkToRootCycle() throws IOException {
        ClassLoader loader = ClassPathTest.class.getClassLoader();
        Path root = Files.createTempDirectory("ClassPathTest", new FileAttribute[0]);
        try {
            Files.createFile(root.resolve("some.txt"), new FileAttribute[0]);
            Path child = Files.createDirectory(root.resolve("child"), new FileAttribute[0]);
            Files.createSymbolicLink(child.resolve("grandchild"), root, new FileAttribute[0]);
            ClassPath.DefaultScanner scanner = new ClassPath.DefaultScanner();
            scanner.scan(root.toFile(), loader);
            ClassPathTest.assertEquals((Object)ImmutableSet.of((Object)new ClassPath.ResourceInfo("some.txt", loader)), (Object)scanner.getResources());
        }
        finally {
            ClassPathTest.deleteRecursivelyOrLog(root);
        }
    }

    public void testScanFromFile_fileNotExists() throws IOException {
        ClassLoader classLoader = ClassPathTest.class.getClassLoader();
        ClassPath.DefaultScanner scanner = new ClassPath.DefaultScanner();
        scanner.scan(new File("no/such/file/anywhere"), classLoader);
        Truth.assertThat((Iterable)scanner.getResources()).isEmpty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testScanFromFile_notJarFile() throws IOException {
        ClassLoader classLoader = ClassPathTest.class.getClassLoader();
        File notJar = File.createTempFile("not_a_jar", "txt");
        ClassPath.DefaultScanner scanner = new ClassPath.DefaultScanner();
        try {
            scanner.scan(notJar, classLoader);
        }
        finally {
            notJar.delete();
        }
        Truth.assertThat((Iterable)scanner.getResources()).isEmpty();
    }

    public void testGetClassPathEntry() throws MalformedURLException, URISyntaxException {
        ClassPathTest.assertEquals((Object)new File("/usr/test/dep.jar").toURI(), (Object)ClassPath.Scanner.getClassPathEntry((File)new File("/home/build/outer.jar"), (String)"file:/usr/test/dep.jar").toURI());
        ClassPathTest.assertEquals((Object)new File("/home/build/a.jar").toURI(), (Object)ClassPath.Scanner.getClassPathEntry((File)new File("/home/build/outer.jar"), (String)"a.jar").toURI());
        ClassPathTest.assertEquals((Object)new File("/home/build/x/y/z").toURI(), (Object)ClassPath.Scanner.getClassPathEntry((File)new File("/home/build/outer.jar"), (String)"x/y/z").toURI());
        ClassPathTest.assertEquals((Object)new File("/home/build/x/y/z.jar").toURI(), (Object)ClassPath.Scanner.getClassPathEntry((File)new File("/home/build/outer.jar"), (String)"x/y/z.jar").toURI());
        ClassPathTest.assertEquals((String)"/home/build/x y.jar", (String)ClassPath.Scanner.getClassPathEntry((File)new File("/home/build/outer.jar"), (String)"x y.jar").getFile());
    }

    public void testGetClassPathFromManifest_nullManifest() {
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)new File("some.jar"), null)).isEmpty();
    }

    public void testGetClassPathFromManifest_noClassPath() throws IOException {
        File jarFile = new File("base.jar");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)ClassPathTest.manifest(""))).isEmpty();
    }

    public void testGetClassPathFromManifest_emptyClassPath() throws IOException {
        File jarFile = new File("base.jar");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)ClassPathTest.manifestClasspath(""))).isEmpty();
    }

    public void testGetClassPathFromManifest_badClassPath() throws IOException {
        File jarFile = new File("base.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("nosuchscheme:an_invalid^path");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).isEmpty();
    }

    public void testGetClassPathFromManifest_pathWithStrangeCharacter() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("file:the^file.jar");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("base/the^file.jar")});
    }

    public void testGetClassPathFromManifest_relativeDirectory() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("with/relative/dir");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("base/with/relative/dir")});
    }

    public void testGetClassPathFromManifest_relativeJar() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("with/relative.jar");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("base/with/relative.jar")});
    }

    public void testGetClassPathFromManifest_jarInCurrentDirectory() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("current.jar");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("base/current.jar")});
    }

    public void testGetClassPathFromManifest_absoluteDirectory() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("file:/with/absolute/dir");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("/with/absolute/dir")});
    }

    public void testGetClassPathFromManifest_absoluteJar() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("file:/with/absolute.jar");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("/with/absolute.jar")});
    }

    public void testGetClassPathFromManifest_multiplePaths() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("file:/with/absolute.jar relative.jar  relative/dir");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("/with/absolute.jar"), ClassPathTest.fullpath("base/relative.jar"), ClassPathTest.fullpath("base/relative/dir")}).inOrder();
    }

    public void testGetClassPathFromManifest_leadingBlanks() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath(" relative.jar");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("base/relative.jar")});
    }

    public void testGetClassPathFromManifest_trailingBlanks() throws IOException {
        File jarFile = new File("base/some.jar");
        Manifest manifest = ClassPathTest.manifestClasspath("relative.jar ");
        Truth.assertThat((Iterable)ClassPath.Scanner.getClassPathFromManifest((File)jarFile, (Manifest)manifest)).containsExactly(new Object[]{ClassPathTest.fullpath("base/relative.jar")});
    }

    public void testGetClassName() {
        ClassPathTest.assertEquals((String)"abc.d.Abc", (String)ClassPath.getClassName((String)"abc/d/Abc.class"));
    }

    public void testResourceInfo_of() {
        ClassPathTest.assertEquals(ClassPath.ClassInfo.class, ClassPathTest.resourceInfo(ClassPathTest.class).getClass());
        ClassPathTest.assertEquals(ClassPath.ClassInfo.class, ClassPathTest.resourceInfo(ClassPath.class).getClass());
        ClassPathTest.assertEquals(ClassPath.ClassInfo.class, ClassPathTest.resourceInfo(Nested.class).getClass());
    }

    public void testGetSimpleName() {
        ClassLoader classLoader = ((Object)((Object)this)).getClass().getClassLoader();
        ClassPathTest.assertEquals((String)"Foo", (String)new ClassPath.ClassInfo("Foo.class", classLoader).getSimpleName());
        ClassPathTest.assertEquals((String)"Foo", (String)new ClassPath.ClassInfo("a/b/Foo.class", classLoader).getSimpleName());
        ClassPathTest.assertEquals((String)"Foo", (String)new ClassPath.ClassInfo("a/b/Bar$Foo.class", classLoader).getSimpleName());
        ClassPathTest.assertEquals((String)"", (String)new ClassPath.ClassInfo("a/b/Bar$1.class", classLoader).getSimpleName());
        ClassPathTest.assertEquals((String)"Foo", (String)new ClassPath.ClassInfo("a/b/Bar$Foo.class", classLoader).getSimpleName());
        ClassPathTest.assertEquals((String)"", (String)new ClassPath.ClassInfo("a/b/Bar$1.class", classLoader).getSimpleName());
        ClassPathTest.assertEquals((String)"Local", (String)new ClassPath.ClassInfo("a/b/Bar$1Local.class", classLoader).getSimpleName());
    }

    public void testGetPackageName() {
        ClassPathTest.assertEquals((String)"", (String)new ClassPath.ClassInfo("Foo.class", ((Object)((Object)this)).getClass().getClassLoader()).getPackageName());
        ClassPathTest.assertEquals((String)"a.b", (String)new ClassPath.ClassInfo("a/b/Foo.class", ((Object)((Object)this)).getClass().getClassLoader()).getPackageName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testGetClassPathUrls() throws Exception {
        String oldPathSeparator = StandardSystemProperty.PATH_SEPARATOR.value();
        String oldClassPath = StandardSystemProperty.JAVA_CLASS_PATH.value();
        System.setProperty(StandardSystemProperty.PATH_SEPARATOR.key(), ":");
        System.setProperty(StandardSystemProperty.JAVA_CLASS_PATH.key(), Joiner.on((String)":").join((Object)"relative/path/to/some.jar", (Object)"/absolute/path/to/some.jar", new Object[]{"relative/path/to/class/root", "/absolute/path/to/class/root"}));
        try {
            ImmutableList urls = ClassPath.Scanner.parseJavaClassPath();
            Truth.assertThat((String)((URL)urls.get(0)).getProtocol()).isEqualTo((Object)"file");
            Truth.assertThat((String)((URL)urls.get(0)).getAuthority()).isNull();
            Truth.assertThat((String)((URL)urls.get(0)).getPath()).endsWith("/relative/path/to/some.jar");
            Truth.assertThat((Object)urls.get(1)).isEqualTo((Object)new URL("file:///absolute/path/to/some.jar"));
            Truth.assertThat((String)((URL)urls.get(2)).getProtocol()).isEqualTo((Object)"file");
            Truth.assertThat((String)((URL)urls.get(2)).getAuthority()).isNull();
            Truth.assertThat((String)((URL)urls.get(2)).getPath()).endsWith("/relative/path/to/class/root");
            Truth.assertThat((Object)urls.get(3)).isEqualTo((Object)new URL("file:///absolute/path/to/class/root"));
            Truth.assertThat((Iterable)urls).hasSize(4);
        }
        finally {
            System.setProperty(StandardSystemProperty.PATH_SEPARATOR.key(), oldPathSeparator);
            System.setProperty(StandardSystemProperty.JAVA_CLASS_PATH.key(), oldClassPath);
        }
    }

    private static boolean contentEquals(URL left, URL right) throws IOException {
        return Resources.asByteSource((URL)left).contentEquals(Resources.asByteSource((URL)right));
    }

    public void testNulls() throws IOException {
        new NullPointerTester().testAllPublicStaticMethods(ClassPath.class);
        new NullPointerTester().testAllPublicInstanceMethods((Object)ClassPath.from((ClassLoader)((Object)((Object)this)).getClass().getClassLoader()));
    }

    public void testResourceScanner() throws IOException {
        ResourceScanner scanner = new ResourceScanner();
        scanner.scan(ClassLoader.getSystemClassLoader());
        Truth.assertThat(scanner.resources).contains((Object)"com/google/common/reflect/ClassPathTest.class");
    }

    public void testExistsThrowsSecurityException() throws IOException, URISyntaxException {
        SecurityManager oldSecurityManager = System.getSecurityManager();
        try {
            this.doTestExistsThrowsSecurityException();
        }
        finally {
            System.setSecurityManager(oldSecurityManager);
        }
    }

    private void doTestExistsThrowsSecurityException() throws IOException, URISyntaxException {
        File file = null;
        final PermissionCollection readClassPathFiles = new FilePermission("", "read").newPermissionCollection();
        for (URL url : ClassPath.Scanner.parseJavaClassPath()) {
            if (!url.getProtocol().equalsIgnoreCase("file")) continue;
            file = new File(url.toURI());
            readClassPathFiles.add(new FilePermission(file.getAbsolutePath(), "read"));
        }
        Truth.assertThat(file).isNotNull();
        SecurityManager disallowFilesSecurityManager = new SecurityManager(){

            @Override
            public void checkPermission(Permission p) {
                if (readClassPathFiles.implies(p)) {
                    throw new SecurityException("Disallowed: " + p);
                }
            }
        };
        System.setSecurityManager(disallowFilesSecurityManager);
        try {
            file.exists();
            ClassPathTest.fail((String)"Did not get expected SecurityException");
        }
        catch (SecurityException url) {
            // empty catch block
        }
        ClassPath classPath = ClassPath.from((ClassLoader)((Object)((Object)this)).getClass().getClassLoader());
        for (ClassPath.ResourceInfo resource : classPath.getResources()) {
            Truth.assertThat((String)resource.getResourceName()).doesNotContain((CharSequence)"com/google/common/reflect/");
        }
    }

    private static ClassPath.ClassInfo findClass(Iterable<ClassPath.ClassInfo> classes, Class<?> cls) {
        for (ClassPath.ClassInfo classInfo : classes) {
            if (!classInfo.getName().equals(cls.getName())) continue;
            return classInfo;
        }
        throw new AssertionError((Object)("failed to find " + cls));
    }

    private static ClassPath.ResourceInfo resourceInfo(Class<?> cls) {
        String resource = cls.getName().replace('.', '/') + ".class";
        ClassLoader loader = cls.getClassLoader();
        return ClassPath.ResourceInfo.of((String)resource, (ClassLoader)loader);
    }

    private static ClassPath.ClassInfo classInfo(Class<?> cls) {
        return ClassPathTest.classInfo(cls, cls.getClassLoader());
    }

    private static ClassPath.ClassInfo classInfo(Class<?> cls, ClassLoader classLoader) {
        String resource = cls.getName().replace('.', '/') + ".class";
        return new ClassPath.ClassInfo(resource, classLoader);
    }

    private static Manifest manifestClasspath(String classpath) throws IOException {
        return ClassPathTest.manifest("Class-Path: " + classpath + "\n");
    }

    private static void writeSelfReferencingJarFile(File jarFile, String ... entries) throws IOException {
        Manifest manifest = new Manifest();
        manifest.getMainAttributes().put(Attributes.Name.MANIFEST_VERSION, "1.0");
        manifest.getMainAttributes().put(Attributes.Name.CLASS_PATH, jarFile.getName());
        try (Closer closer = Closer.create();){
            FileOutputStream fileOut = (FileOutputStream)closer.register((Closeable)new FileOutputStream(jarFile));
            JarOutputStream jarOut = (JarOutputStream)closer.register((Closeable)new JarOutputStream(fileOut));
            for (String entry : entries) {
                jarOut.putNextEntry(new ZipEntry(entry));
                Resources.copy((URL)ClassPathTest.class.getResource(entry), (OutputStream)jarOut);
                jarOut.closeEntry();
            }
        }
    }

    private static Manifest manifest(String content) throws IOException {
        ByteArrayInputStream in = new ByteArrayInputStream(content.getBytes(Charsets.US_ASCII));
        Manifest manifest = new Manifest();
        manifest.read(in);
        return manifest;
    }

    private static File fullpath(String path) {
        return new File(new File(path).toURI());
    }

    private static URL makeJarUrlWithName(String name) throws IOException {
        File fullPath = new File(com.google.common.io.Files.createTempDir(), name);
        File jarFile = JarFileFinder.pickAnyJarFile();
        com.google.common.io.Files.copy((File)jarFile, (File)fullPath);
        return fullPath.toURI().toURL();
    }

    @AndroidIncompatible
    private static void deleteRecursivelyOrLog(Path path) {
        try {
            MoreFiles.deleteRecursively((Path)path, (RecursiveDeleteOption[])new RecursiveDeleteOption[0]);
        }
        catch (IOException e) {
            log.log(Level.WARNING, "Failure cleaning up test directory", e);
        }
    }

    private static final class JarFileFinder
    extends ClassPath.Scanner {
        private File found;

        private JarFileFinder() {
        }

        static File pickAnyJarFile() throws IOException {
            JarFileFinder finder = new JarFileFinder();
            try {
                finder.scan(JarFileFinder.class.getClassLoader());
                throw new IllegalStateException("No jar file found!");
            }
            catch (StopScanningException expected) {
                return finder.found;
            }
        }

        protected void scanJarFile(ClassLoader loader, JarFile file) throws IOException {
            this.found = new File(file.getName());
            throw new StopScanningException();
        }

        protected void scanDirectory(ClassLoader loader, File root) {
        }

        private static final class StopScanningException
        extends RuntimeException {
            private StopScanningException() {
            }
        }
    }

    private static class ResourceScanner
    extends ClassPath.Scanner {
        final Set<String> resources = new HashSet<String>();

        private ResourceScanner() {
        }

        protected void scanDirectory(ClassLoader loader, File root) throws IOException {
            URI base = root.toURI();
            for (File entry : com.google.common.io.Files.fileTreeTraverser().preOrderTraversal((Object)root)) {
                String resourceName = new File(base.relativize(entry.toURI()).getPath()).getPath();
                this.resources.add(resourceName);
            }
        }

        protected void scanJarFile(ClassLoader loader, JarFile file) throws IOException {
            Enumeration<JarEntry> entries = file.entries();
            while (entries.hasMoreElements()) {
                this.resources.add(entries.nextElement().getName());
            }
        }
    }

    private static class Nested {
        private Nested() {
        }
    }
}

