/*
 * Copyright 2020 The Closure Compiler Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.javascript.jscomp.colors;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.ImmutableSet.toImmutableSet;

import com.google.auto.value.AutoValue;
import com.google.auto.value.extension.memoized.Memoized;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import java.util.Set;
import javax.annotation.Nullable;

/** A simplified version of a Closure or TS type for use by optimizations */
@AutoValue
public abstract class Color {

  public abstract ColorId getId();

  public abstract DebugInfo getDebugInfo();

  /** Given `function Foo() {}` or `class Foo {}`, color of Foo.prototype. null otherwise. */
  public abstract ImmutableSet<Color> getPrototypes();

  public abstract ImmutableSet<Color> getInstanceColors();

  /**
   * List of other colors directly above this in the subtyping graph for the purposes of property
   * (dis)ambiguation.
   */
  public abstract ImmutableSet<Color> getDisambiguationSupertypes();

  public abstract boolean isInvalidating();

  public abstract boolean getPropertiesKeepOriginalName();

  public abstract boolean isConstructor();

  /**
   * Property names 'declared' on an object (as opposed to being conceptually inherited from some
   * supertype).
   */
  public abstract ImmutableSet<String> getOwnProperties();

  @Nullable
  public abstract NativeColorId getNativeColorId();

  /**
   * Whether this type is some Closure assertion function removable by Closure-specific
   * optimizations.
   */
  public abstract boolean isClosureAssert();

  public abstract ImmutableSet<Color> getUnionElements();

  public static Builder singleBuilder() {
    return new AutoValue_Color.Builder()
        .setClosureAssert(false)
        .setConstructor(false)
        .setDebugInfo(DebugInfo.EMPTY)
        .setDisambiguationSupertypes(ImmutableSet.of())
        .setInstanceColors(ImmutableSet.of())
        .setInvalidating(false)
        .setOwnProperties(ImmutableSet.of())
        .setPropertiesKeepOriginalName(false)
        .setPrototypes(ImmutableSet.of())
        .setUnionElements(ImmutableSet.of());
  }

  public static Color createUnion(Set<Color> elements) {
    switch (elements.size()) {
      case 0:
        throw new IllegalStateException();
      case 1:
        return Iterables.getOnlyElement(elements);
      default:
        break;
    }

    ImmutableSet.Builder<Color> disambiguationSupertypes = ImmutableSet.builder();
    ImmutableSet.Builder<Color> instanceColors = ImmutableSet.builder();
    ImmutableSet.Builder<Color> prototypes = ImmutableSet.builder();
    ImmutableSet.Builder<Color> newElements = ImmutableSet.builder();
    ImmutableSet.Builder<ColorId> ids = ImmutableSet.builder();
    ImmutableSet.Builder<String> ownProperties = ImmutableSet.builder();
    boolean isClosureAssert = true;
    boolean isConstructor = true;
    boolean isInvalidating = false;
    boolean propertiesKeepOriginalName = false;

    for (Color element : elements) {
      if (element.isUnion()) {
        for (Color nestedElement : element.getUnionElements()) {
          newElements.add(nestedElement);
          ids.add(nestedElement.getId());
        }
      } else {
        newElements.add(element);
        ids.add(element.getId());
      }

      disambiguationSupertypes.addAll(element.getDisambiguationSupertypes());
      instanceColors.addAll(element.getInstanceColors());
      isClosureAssert &= element.isClosureAssert();
      isConstructor &= element.isConstructor();
      isInvalidating |= element.isInvalidating();
      ownProperties.addAll(element.getOwnProperties()); // Are these actually the "own props"?
      propertiesKeepOriginalName |= element.getPropertiesKeepOriginalName();
      prototypes.addAll(element.getPrototypes());
    }

    return new AutoValue_Color.Builder()
        .setClosureAssert(isClosureAssert)
        .setConstructor(isConstructor)
        .setDebugInfo(DebugInfo.EMPTY)
        .setDisambiguationSupertypes(disambiguationSupertypes.build())
        .setId(ColorId.union(ids.build()))
        .setInstanceColors(instanceColors.build())
        .setInvalidating(isInvalidating)
        .setOwnProperties(ownProperties.build())
        .setPropertiesKeepOriginalName(propertiesKeepOriginalName)
        .setPrototypes(prototypes.build())
        .setUnionElements(newElements.build())
        .buildUnion();
  }

  Color() {
    // No public constructor.
  }

  /**
   * Whether this corresponds to a single JavaScript primitive like number or symbol.
   *
   * <p>Note that the boxed versions of primitives (String, Number, etc.) are /not/ considered
   * "primitive" by this method.
   */
  public final boolean isPrimitive() {
    /**
     * Avoid the design headache about whether unions are primitives. The union *color* isn't
     * primitive, but the *value* held by a union reference may be.
     */
    checkState(!this.isUnion(), this);
    return this.getNativeColorId() != null && this.getNativeColorId().isPrimitive();
  }

  public final boolean isUnion() {
    // Single element sets are banned in the builder.
    return !this.getUnionElements().isEmpty();
  }

  /** Whether this is exactly the given native color (and not a union containing that color). */
  public final boolean is(NativeColorId color) {
    return this.getNativeColorId() == color;
  }

  /**
   * Returns true if the color or any of its ancestors has the given property
   *
   * <p>If this is a union, returns true if /any/ union alternate has the property.
   *
   * <p>TODO(b/177695515): delete this method
   */
  public boolean mayHaveProperty(String propertyName) {
    if (this.isUnion()) {
      return this.getUnionElements().stream()
          .anyMatch(element -> element.mayHaveProperty(propertyName));
    }

    if (this.getOwnProperties().contains(propertyName)) {
      return true;
    }
    return this.getDisambiguationSupertypes().stream()
        .anyMatch(element -> element.mayHaveProperty(propertyName));
  }

  @Memoized
  public Color subtractNullOrVoid() {
    /**
     * Forbid calling this on non-unions to avoid defining what NULL_OR_VOID.subtract(NULL_OR_VOID)
     * is.
     */
    ImmutableSet<Color> elements =
        this.getUnionElements().stream()
            .filter(alt -> !alt.is(NativeColorId.NULL_OR_VOID))
            .collect(toImmutableSet());
    return (elements.size() == this.getUnionElements().size()) ? this : createUnion(elements);
  }

  /**
   * Builder for a singleton color. Should be passed to {@link
   * Color#createSingleton(SingletonColorFields)} after building and before using
   */
  @AutoValue.Builder
  public abstract static class Builder {

    public abstract Builder setId(ColorId x);

    public abstract Builder setInvalidating(boolean x);

    public abstract Builder setPropertiesKeepOriginalName(boolean x);

    public abstract Builder setDisambiguationSupertypes(ImmutableSet<Color> x);

    public abstract Builder setConstructor(boolean x);

    public abstract Builder setOwnProperties(ImmutableSet<String> x);

    public abstract Builder setDebugInfo(DebugInfo x);

    public abstract Builder setClosureAssert(boolean x);

    abstract Builder setNativeColorId(@Nullable NativeColorId x);

    abstract Builder setPrototypes(ImmutableSet<Color> x);

    abstract Builder setInstanceColors(ImmutableSet<Color> x);

    abstract Builder setUnionElements(ImmutableSet<Color> x);

    @VisibleForTesting
    public Builder setDebugName(String x) {
      return this.setDebugInfo(DebugInfo.builder().setClassName(x).build());
    }

    public Builder setPrototype(Color x) {
      return this.setPrototypes((x == null) ? ImmutableSet.of() : ImmutableSet.of(x));
    }

    public Builder setInstanceColor(Color x) {
      return this.setInstanceColors((x == null) ? ImmutableSet.of() : ImmutableSet.of(x));
    }

    abstract Color buildInternal();

    public final Color build() {
      Color result = this.buildInternal();
      checkState(result.getUnionElements().isEmpty(), result);
      return result;
    }

    @SuppressWarnings("ReferenceEquality")
    private final Color buildUnion() {
      Color result = this.buildInternal();
      checkState(result.getUnionElements().size() > 1, result);
      checkState(result.getDebugInfo() == DebugInfo.EMPTY, result);
      checkState(result.getNativeColorId() == null, result);
      return result;
    }
  }
}
