# JsSrc design

The jssrc backend is in the middle of a migration between JsCodeBuilder+JsExpr
-> CodeChunk+CodeChunk.WithValue for managing code generation. This will strive
to document some of the current design of the backend in order to inform future
refactorings.

## GenJsCodeVisitor

This class is the main entry point for code generation, it handles

*   file level declaration
*   template declarations
*   all SoyNode constructs

In order to do this it uses a number of collaborators. The key ones for
understanding the overall strategy are:

*   GenJsExprsVisitor: a SoyNodeVisitor that can translate a subset of SoyNodes
    to `CodeChunk.WithValue` objects.
*   IsComputableAsJsExprsVisitor: a SoyNodeVisitor that can tell whether or a
    SoyNode can be handled by `GenJsExprsVisitor`
*   CanInitOutputVarsVisitor: a SoyNodeVisitor that can tell whether a given
    SoyNode will evaluate to a single string typed value and thus can be used to
    initialize an output variable. The main point of this is to ensure that
    output variables are declared in the appropriate block scope (i.e. not
    inside a loop or conditional block).
*   JsCodeBuilder: the main output buffer. In addition to recording all the
    generated code it also manages the state of a set of 'output variables'. An
    output variable is a synthetic variable generated by the compiler that is
    used to accumulate template output.

So right off the bat we can see that there is some redundancy between the
visitors. e.g `IsComputableAsJsExprsVisitor` and `GenJsExprsVisitor` must be
kept in sync to work correctly. Similarly `CanInitiOutputVarsVisitor` needs to
be kept in sync with `GenJsExprsVisitor`.

This complex strategy is used to ensure that we:

*   declare as few output variables as possible
*   prefer generating code that looks like `output += a + b + c` instead of
    `output += a; output += b; output += c`

This is because it allows the jscompiler to generate smaller code (This in turn
is because lots of our statements tend to have declarations associated with them
and the jscompiler is generally unwilling to reorder those appends with
arbitrary method calls). For example, while we know that:

```js
var output = '' + a + b;
var MSG_XXX = goog.getMsg(....);
output += MSG_XXX;
```

is equivalent to

```js
var MSG_XXX = goog.getMsg(....);
var output = '' + a + b + MSG_XXX;
```

The jscompiler isn't sure that it is safe to reorder the declarations even though
the second implementation is shorter.

### Path to eliminating `JsCodeBuilder`

We have a long term goal to eliminate `JsCodeBuilder` and instead deal entirely
with `CodeChunks`. This is difficult however if we want to maintain the goals
stated above. So a few concrete steps to take would be:

*   Move the logic for file and template declarations from GenJsCodeVisitor to
    another class, let GenJsCodeVisitor focus on template bodies. This would be
    a straight refactoring.
*   Introduce some kind of datastructure like `AppendsOrConcats` which would
    either be a `CodeChunk` that assigns appends to a declared output variable,
    or a sequence of `CodeChunk.WithValue` objects that evaluate to strings.
    *   We may want to be able to lazily pick the output variable, so we may
        need something like a `Function<CodeChunk.WithValue, CodeChunk>` instead
        of just a `CodeChunk`
*   Change `GenJsCodeVisitor` to return `AppendsOrConcats` objects instead of
    appending to the `JsCodeBuilder`, at this point we would only be using the
    'appends' half of the datastructure.
*   Move the logic of `GenJsExprsVisitor` into `GenJsCodeVisitor`. This way
    instead of using `IsComputableAsJsExprsVisitor` to decide whether we should
    use `GenJsCodeVisitor` or `GenJsExprsVisitor` for a given expression, it
    would just become if statements inside the `visit` methods.

Some of these steps might be pretty difficult, but it should lead us to a
solution that is easier to understand, relies less on mutable datastructures and
it should actually allow us to generate less code.
