/*
 * Copyright 2015 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.template.soy.shared.internal;

import com.google.template.soy.data.SanitizedContent;
import com.google.template.soy.data.SoyMap;
import com.google.template.soy.data.SoyRecord;
import com.google.template.soy.data.SoyValue;
import com.google.template.soy.data.SoyValueProvider;
import com.google.template.soy.data.internal.SoyMapImpl;
import com.google.template.soy.data.restricted.FloatData;
import com.google.template.soy.data.restricted.IntegerData;
import com.google.template.soy.data.restricted.NumberData;
import com.google.template.soy.data.restricted.StringData;
import com.google.template.soy.exprtree.MapLiteralFromListNode;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.OptionalInt;

/**
 * Runtime implementation of common expression operators to be shared between the {@code jbcsrc} and
 * {@code Tofu} backends.
 */
public final class SharedRuntime {
  /**
   * Custom equality operator that smooths out differences between different Soy runtimes.
   *
   * <p>This approximates Javascript's behavior, but is much easier to understand.
   */
  public static boolean equal(SoyValue operand0, SoyValue operand1) {
    // Treat the case where either is a string specially.
    // TODO(gboyer): This should probably handle SanitizedContent == SanitizedContent, even though
    // Javascript doesn't handle that case properly. http://b/21461181
    if (operand0 instanceof StringData) {
      return compareString(operand0.stringValue(), operand1);
    }
    if (operand1 instanceof StringData) {
      return compareString(operand1.stringValue(), operand0);
    }
    return Objects.equals(operand0, operand1);
  }

  /** Performs the {@code +} operator on the two values. */
  public static SoyValue plus(SoyValue operand0, SoyValue operand1) {
    if (operand0 instanceof IntegerData && operand1 instanceof IntegerData) {
      return IntegerData.forValue(operand0.longValue() + operand1.longValue());
    } else if (operand0 instanceof NumberData && operand1 instanceof NumberData) {
      return FloatData.forValue(operand0.numberValue() + operand1.numberValue());
    } else {
      // String concatenation is the fallback for other types (like in JS). Use the implemented
      // coerceToString() for the type.
      return StringData.forValue(operand0.coerceToString() + operand1.coerceToString());
    }
  }

  /** Performs the {@code -} operator on the two values. */
  public static SoyValue minus(SoyValue operand0, SoyValue operand1) {
    if (operand0 instanceof IntegerData && operand1 instanceof IntegerData) {
      return IntegerData.forValue(operand0.longValue() - operand1.longValue());
    } else {
      return FloatData.forValue(operand0.numberValue() - operand1.numberValue());
    }
  }

  /** Performs the {@code *} operator on the two values. */
  public static NumberData times(SoyValue operand0, SoyValue operand1) {
    if (operand0 instanceof IntegerData && operand1 instanceof IntegerData) {
      return IntegerData.forValue(operand0.longValue() * operand1.longValue());
    } else {
      return FloatData.forValue(operand0.numberValue() * operand1.numberValue());
    }
  }

  /** Performs the {@code /} operator on the two values. */
  public static double dividedBy(SoyValue operand0, SoyValue operand1) {
    // Note: Soy always performs floating-point division, even on two integers (like JavaScript).
    // Note that this *will* lose precision for longs.
    return operand0.numberValue() / operand1.numberValue();
  }
  /** Performs the {@code %} operator on the two values. */
  public static NumberData mod(SoyValue operand0, SoyValue operand1) {
    if (operand0 instanceof IntegerData && operand1 instanceof IntegerData) {
      return IntegerData.forValue(operand0.longValue() % operand1.longValue());
    } else {
      return FloatData.forValue(operand0.numberValue() % operand1.numberValue());
    }
  }

  public static NumberData shiftRight(SoyValue operand0, SoyValue operand1) {
    return IntegerData.forValue(operand0.longValue() >> (int) operand1.longValue());
  }

  public static NumberData shiftLeft(SoyValue operand0, SoyValue operand1) {
    return IntegerData.forValue(operand0.longValue() << (int) operand1.longValue());
  }

  public static NumberData bitwiseOr(SoyValue operand0, SoyValue operand1) {
    return IntegerData.forValue(operand0.longValue() | operand1.longValue());
  }

  public static NumberData bitwiseXor(SoyValue operand0, SoyValue operand1) {
    return IntegerData.forValue(operand0.longValue() ^ operand1.longValue());
  }

  public static NumberData bitwiseAnd(SoyValue operand0, SoyValue operand1) {
    return IntegerData.forValue(operand0.longValue() & operand1.longValue());
  }

  /** Performs the {@code <} operator on the two values. */
  public static boolean lessThan(SoyValue left, SoyValue right) {
    if (left instanceof StringData && right instanceof StringData) {
      return left.stringValue().compareTo(right.stringValue()) < 0;
    } else if (left instanceof IntegerData && right instanceof IntegerData) {
      return left.longValue() < right.longValue();
    } else {
      return left.numberValue() < right.numberValue();
    }
  }

  /** Performs the {@code <=} operator on the two values. */
  public static boolean lessThanOrEqual(SoyValue left, SoyValue right) {
    if (left instanceof StringData && right instanceof StringData) {
      return left.stringValue().compareTo(right.stringValue()) <= 0;
    } else if (left instanceof IntegerData && right instanceof IntegerData) {
      return left.longValue() <= right.longValue();
    } else {
      return left.numberValue() <= right.numberValue();
    }
  }

  /** Performs the unary negation {@code -} operator on the value. */
  public static NumberData negative(SoyValue node) {
    if (node instanceof IntegerData) {
      return IntegerData.forValue(-node.longValue());
    } else {
      return FloatData.forValue(-node.floatValue());
    }
  }

  /** Determines if the operand's string form can be equality-compared with a string. */
  public static boolean compareString(String string, SoyValue other) {
    // This follows similarly to the Javascript specification, to ensure similar operation
    // over Javascript and Java: http://www.ecma-international.org/ecma-262/5.1/#sec-11.9.3
    if (other instanceof StringData || other instanceof SanitizedContent) {
      return string.equals(other.toString());
    }
    if (other instanceof NumberData) {
      try {
        // Parse the string as a number.
        return Double.parseDouble(string) == other.numberValue();
      } catch (NumberFormatException nfe) {
        // Didn't parse as a number.
        return false;
      }
    }
    return false;
  }

  /** calculates a $soyServerKey value. This should be compatible with the JS implementation in */
  public static String soyServerKey(SoyValue key) {
    if (key instanceof NumberData) {
      return serialize(key.coerceToString(), "#");
    }
    if (key == null) {
      return serialize("null", "_");
    }
    return serialize(key.coerceToString(), ":");
  }

  public static SoyMap constructMapFromList(List<? extends SoyValueProvider> list) {
    Map<SoyValue, SoyValueProvider> map = new HashMap<>();
    for (int i = 0; i < list.size(); i++) {
      SoyValue recordEntry = list.get(i).resolve();
      checkMapFromListConstructorCondition(
          recordEntry instanceof SoyRecord, recordEntry, OptionalInt.of(i));

      checkMapFromListConstructorCondition(
          ((SoyRecord) recordEntry).hasField(MapLiteralFromListNode.KEY_STRING)
              && ((SoyRecord) recordEntry).hasField(MapLiteralFromListNode.VALUE_STRING),
          recordEntry,
          OptionalInt.of(i));

      SoyValue key = ((SoyRecord) recordEntry).getField(MapLiteralFromListNode.KEY_STRING);
      SoyValueProvider valueProvider =
          ((SoyRecord) recordEntry).getFieldProvider(MapLiteralFromListNode.VALUE_STRING);
      checkMapFromListConstructorCondition(
          key != null && SoyMap.isAllowedKeyType(key), recordEntry, OptionalInt.of(i));

      map.put(key, valueProvider);
    }

    return SoyMapImpl.forProviderMap(map);
  }

  public static void checkMapFromListConstructorCondition(
      boolean condition, SoyValue list, OptionalInt index) {
    if (!condition) {
      String exceptionString =
          String.format(
              "Error constructing map. Expected a list where each item is a record of 'key',"
                  + " 'value' pairs, with the 'key' fields holding primitive values. Found: %s",
              list);
      if (index.isPresent()) {
        exceptionString += String.format(" at index %d", index.getAsInt());
      }

      // TODO: throw a RenderException here
      throw new IllegalArgumentException(exceptionString);
    }
  }

  private static String serialize(String key, String delimiter) {
    return key.length() + delimiter + key;
  }

  private SharedRuntime() {}
}
