/*
 * Decompiled with CFR 0.152.
 */
package com.gotocompany.depot.bigquery.client;

import com.google.auth.Credentials;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.cloud.TransportOptions;
import com.google.cloud.bigquery.BigQuery;
import com.google.cloud.bigquery.BigQueryException;
import com.google.cloud.bigquery.BigQueryOptions;
import com.google.cloud.bigquery.Clustering;
import com.google.cloud.bigquery.Dataset;
import com.google.cloud.bigquery.Field;
import com.google.cloud.bigquery.InsertAllRequest;
import com.google.cloud.bigquery.InsertAllResponse;
import com.google.cloud.bigquery.Schema;
import com.google.cloud.bigquery.StandardTableDefinition;
import com.google.cloud.bigquery.Table;
import com.google.cloud.bigquery.TableDefinition;
import com.google.cloud.bigquery.TableId;
import com.google.cloud.bigquery.TableInfo;
import com.google.cloud.bigquery.TimePartitioning;
import com.google.cloud.http.HttpTransportOptions;
import com.gotocompany.depot.bigquery.client.BigQueryTableDefinition;
import com.gotocompany.depot.bigquery.exception.BQDatasetLocationChangedException;
import com.gotocompany.depot.config.BigQuerySinkConfig;
import com.gotocompany.depot.metrics.BigQueryMetrics;
import com.gotocompany.depot.metrics.Instrumentation;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.Instant;
import java.util.List;
import java.util.Random;

public class BigQueryClient {
    private static final int TABLE_INFO_UPDATE_RETRIES = 10;
    private static final int DEFAULT_SLEEP_RETRY = 10000;
    private final BigQuery bigquery;
    private final TableId tableID;
    private final BigQuerySinkConfig bqConfig;
    private final BigQueryTableDefinition bigQueryTableDefinition;
    private final Instrumentation instrumentation;
    private final Random random = new Random(System.currentTimeMillis());
    private final BigQueryMetrics bigqueryMetrics;

    public BigQueryClient(BigQuerySinkConfig bqConfig, BigQueryMetrics bigQueryMetrics, Instrumentation instrumentation) throws IOException {
        this(BigQueryClient.getBigQueryInstance(bqConfig), bqConfig, bigQueryMetrics, instrumentation);
    }

    public BigQueryClient(BigQuery bq, BigQuerySinkConfig bqConfig, BigQueryMetrics bigQueryMetrics, Instrumentation instrumentation) {
        this.bigquery = bq;
        this.bqConfig = bqConfig;
        this.tableID = TableId.of((String)bqConfig.getDatasetName(), (String)bqConfig.getTableName());
        this.bigQueryTableDefinition = new BigQueryTableDefinition(bqConfig);
        this.instrumentation = instrumentation;
        this.bigqueryMetrics = bigQueryMetrics;
    }

    private static BigQuery getBigQueryInstance(BigQuerySinkConfig sinkConfig) throws IOException {
        HttpTransportOptions transportOptions = BigQueryOptions.getDefaultHttpTransportOptions().toBuilder().setConnectTimeout(sinkConfig.getBqClientConnectTimeoutMS()).setReadTimeout(sinkConfig.getBqClientReadTimeoutMS()).build();
        return (BigQuery)((BigQueryOptions.Builder)((BigQueryOptions.Builder)BigQueryOptions.newBuilder().setTransportOptions((TransportOptions)transportOptions).setCredentials((Credentials)GoogleCredentials.fromStream((InputStream)new FileInputStream(sinkConfig.getBigQueryCredentialPath())))).setProjectId(sinkConfig.getGCloudProjectID())).build().getService();
    }

    public InsertAllResponse insertAll(InsertAllRequest rows) {
        Instant start = Instant.now();
        InsertAllResponse response = this.bigquery.insertAll(rows);
        this.instrument(start, BigQueryMetrics.BigQueryAPIType.TABLE_INSERT_ALL);
        return response;
    }

    public void upsertTable(List<Field> bqSchemaFields) throws BigQueryException {
        Schema schema = Schema.of(bqSchemaFields);
        TableDefinition tableDefinition = this.getTableDefinition(schema);
        TableInfo tableInfo = TableInfo.newBuilder((TableId)this.tableID, (TableDefinition)tableDefinition).setLabels(this.bqConfig.getTableLabels()).build();
        this.upsertDatasetAndTableWithRetry(tableInfo);
    }

    public Schema getSchema() {
        Table table = this.bigquery.getTable(this.tableID, new BigQuery.TableOption[0]);
        if (table == null) {
            return Schema.of((Field[])new Field[0]);
        }
        return table.getDefinition().getSchema();
    }

    private void upsertDatasetAndTableWithRetry(TableInfo info) {
        for (int ii = 0; ii < 10; ++ii) {
            try {
                this.upsertDatasetAndTable(info);
                return;
            }
            catch (BigQueryException e) {
                this.instrumentation.logWarn(e.getMessage(), new Object[0]);
                if (e.getMessage().contains("Exceeded rate limits")) {
                    try {
                        int sleepMillis = this.random.nextInt(10000);
                        this.instrumentation.logInfo("Waiting for " + sleepMillis + " milliseconds", new Object[0]);
                        Thread.sleep(sleepMillis);
                    }
                    catch (InterruptedException interruptedException) {
                        this.instrumentation.captureNonFatalError(this.bigqueryMetrics.getErrorEventMetric(), interruptedException, "Sleep interrupted", new Object[0]);
                    }
                    continue;
                }
                throw e;
            }
        }
    }

    private void upsertDatasetAndTable(TableInfo tableInfo) {
        Instant start;
        Dataset dataSet = this.bigquery.getDataset(this.tableID.getDataset(), new BigQuery.DatasetOption[0]);
        if (dataSet == null || !this.bigquery.getDataset(this.tableID.getDataset(), new BigQuery.DatasetOption[0]).exists()) {
            start = Instant.now();
            this.bigquery.create(Dataset.newBuilder((String)this.tableID.getDataset()).setLocation(this.bqConfig.getBigQueryDatasetLocation()).setLabels(this.bqConfig.getDatasetLabels()).build(), new BigQuery.DatasetOption[0]);
            this.instrumentation.logInfo("Successfully CREATED bigquery DATASET: {}", this.tableID.getDataset());
            this.instrument(start, BigQueryMetrics.BigQueryAPIType.DATASET_CREATE);
        } else if (this.shouldUpdateDataset(dataSet)) {
            start = Instant.now();
            this.bigquery.update(Dataset.newBuilder((String)this.tableID.getDataset()).setLabels(this.bqConfig.getDatasetLabels()).build(), new BigQuery.DatasetOption[0]);
            this.instrumentation.logInfo("Successfully UPDATED bigquery DATASET: {} with labels", this.tableID.getDataset());
            this.instrument(start, BigQueryMetrics.BigQueryAPIType.DATASET_UPDATE);
        }
        Table table = this.bigquery.getTable(this.tableID, new BigQuery.TableOption[0]);
        if (table == null || !table.exists()) {
            Instant start2 = Instant.now();
            this.bigquery.create(tableInfo, new BigQuery.TableOption[0]);
            this.instrumentation.logInfo("Successfully CREATED bigquery TABLE: {}", this.tableID.getTable());
            this.instrument(start2, BigQueryMetrics.BigQueryAPIType.TABLE_CREATE);
        } else {
            Schema updatedSchema;
            Schema existingSchema = table.getDefinition().getSchema();
            if (this.shouldUpdateTable(tableInfo, table, existingSchema, updatedSchema = tableInfo.getDefinition().getSchema())) {
                Instant start3 = Instant.now();
                this.bigquery.update(tableInfo, new BigQuery.TableOption[0]);
                this.instrumentation.logInfo("Successfully UPDATED bigquery TABLE: {}", this.tableID.getTable());
                this.instrument(start3, BigQueryMetrics.BigQueryAPIType.TABLE_UPDATE);
            } else {
                this.instrumentation.logInfo("Skipping bigquery table update, since proto schema hasn't changed", new Object[0]);
            }
        }
    }

    private void instrument(Instant startTime, BigQueryMetrics.BigQueryAPIType type) {
        this.instrumentation.incrementCounter(this.bigqueryMetrics.getBigqueryOperationTotalMetric(), String.format("table=%s", this.tableID.getTable()), String.format("dataset=%s", this.tableID.getDataset()), String.format("project=%s", this.tableID.getProject()), String.format("api=%s", new Object[]{type}));
        this.instrumentation.captureDurationSince(this.bigqueryMetrics.getBigqueryOperationLatencyMetric(), startTime, String.format("table=%s", this.tableID.getTable()), String.format("dataset=%s", this.tableID.getDataset()), String.format("project=%s", this.tableID.getProject()), String.format("api=%s", new Object[]{type}));
    }

    private boolean shouldUpdateTable(TableInfo tableInfo, Table table, Schema existingSchema, Schema updatedSchema) {
        return !table.getLabels().equals(tableInfo.getLabels()) || !existingSchema.equals((Object)updatedSchema) || this.shouldChangePartitionExpiryForStandardTable(table) || this.shouldUpdateClusteringKeys(table);
    }

    private boolean shouldUpdateDataset(Dataset dataSet) {
        if (!dataSet.getLocation().equals(this.bqConfig.getBigQueryDatasetLocation())) {
            throw new BQDatasetLocationChangedException("Dataset location cannot be changed from " + dataSet.getLocation() + " to " + this.bqConfig.getBigQueryDatasetLocation());
        }
        return !dataSet.getLabels().equals(this.bqConfig.getDatasetLabels());
    }

    private boolean shouldChangePartitionExpiryForStandardTable(Table table) {
        if (!this.isTable(table)) {
            return false;
        }
        TimePartitioning timePartitioning = ((StandardTableDefinition)table.getDefinition()).getTimePartitioning();
        if (timePartitioning == null) {
            return false;
        }
        long neverExpireMs = 0L;
        Long currentExpirationMs = timePartitioning.getExpirationMs() == null ? neverExpireMs : timePartitioning.getExpirationMs();
        Long newExpirationMs = this.bqConfig.getBigQueryTablePartitionExpiryMS() > 0L ? this.bqConfig.getBigQueryTablePartitionExpiryMS() : neverExpireMs;
        return !currentExpirationMs.equals(newExpirationMs);
    }

    private boolean shouldUpdateClusteringKeys(Table table) {
        if (!this.isTable(table)) {
            return false;
        }
        Clustering clustering = ((StandardTableDefinition)table.getDefinition()).getClustering();
        if (clustering != null) {
            List existingClusteringKeys = clustering.getFields();
            if (this.bqConfig.isTableClusteringEnabled().booleanValue()) {
                List<String> updatedClusteringKeys = this.bqConfig.getTableClusteringKeys();
                return !existingClusteringKeys.equals(updatedClusteringKeys);
            }
            return false;
        }
        return this.bqConfig.isTableClusteringEnabled();
    }

    private boolean isTable(Table table) {
        return table.getDefinition().getType().equals((Object)TableDefinition.Type.TABLE);
    }

    private TableDefinition getTableDefinition(Schema schema) {
        return this.bigQueryTableDefinition.getTableDefinition(schema);
    }

    public TableId getTableID() {
        return this.tableID;
    }
}

