package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** Represents a Persistent Disk resource in AWS.

An AWS EBS disk must exist before mounting to a container. The disk must also be in the same AWS zone as the kubelet. An AWS EBS disk can only be mounted as read/write once. AWS EBS volumes support ownership management and SELinux relabeling.
 *
 *  @param  volumeID  volumeID is unique ID of the persistent disk resource in AWS (Amazon EBS volume). More info:
 *                    https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 *  @param    fsType  fsType is the filesystem type of the volume that you want to mount. Tip: Ensure that the filesystem type
 *                    is supported by the host operating system. Examples: "ext4", "xfs", "ntfs". Implicitly inferred to be
 *                    "ext4" if unspecified. More info: https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 *  @param partition  partition is the partition in the volume that you want to mount. If omitted, the default is to mount
 *                    by volume name. Examples: For volume /dev/sda1, you specify the partition as "1". Similarly, the volume
 *                    partition for /dev/sda is "0" (or you can leave the property empty).
 *  @param  readOnly  readOnly value true will force the readOnly setting in VolumeMounts. More info: https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
*/
case class AWSElasticBlockStoreVolumeSource(
  volumeID: String,
  fsType: Option[String] = None,
  partition: Option[Int] = None,
  readOnly: Option[Boolean] = None
)

object AWSElasticBlockStoreVolumeSource {
  implicit lazy val encoder: Encoder.AsObject[AWSElasticBlockStoreVolumeSource] = deriveEncoder
  implicit lazy val decoder: Decoder[AWSElasticBlockStoreVolumeSource] = deriveDecoder
}
