package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** Represents a Fibre Channel volume. Fibre Channel volumes can only be mounted as read/write once. Fibre Channel volumes support ownership management and SELinux relabeling.
 *
 *  @param     fsType  fsType is the filesystem type to mount. Must be a filesystem type supported by the host operating system.
 *                     Ex. "ext4", "xfs", "ntfs". Implicitly inferred to be "ext4" if unspecified.
 *  @param        lun  lun is Optional: FC target lun number
 *  @param   readOnly  readOnly is Optional: Defaults to false (read/write). ReadOnly here will force the ReadOnly setting
 *                     in VolumeMounts.
 *  @param targetWWNs  targetWWNs is Optional: FC target worldwide names (WWNs)
 *  @param      wwids  wwids Optional: FC volume world wide identifiers (wwids) Either wwids or combination of targetWWNs and
 *                     lun must be set, but not both simultaneously.
*/
case class FCVolumeSource(
  fsType: Option[String] = None,
  lun: Option[Int] = None,
  readOnly: Option[Boolean] = None,
  targetWWNs: Option[Seq[String]] = None,
  wwids: Option[Seq[String]] = None
)

object FCVolumeSource {
  implicit lazy val encoder: Encoder.AsObject[FCVolumeSource] = deriveEncoder
  implicit lazy val decoder: Decoder[FCVolumeSource] = deriveDecoder
}
