package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** Lifecycle describes actions that the management system should take in response to container lifecycle events. For the PostStart and PreStop lifecycle handlers, management of the container blocks until the action is complete, unless the container process fails, in which case the handler is aborted.
 *
 *  @param postStart  PostStart is called immediately after a container is created. If the handler fails, the container is
 *                    terminated and restarted according to its restart policy. Other management of the container blocks until
 *                    the hook completes. More info: https://kubernetes.io/docs/concepts/containers/container-lifecycle-hooks/#container-hooks
 *  @param   preStop  PreStop is called immediately before a container is terminated due to an API request or management event
 *                    such as liveness/startup probe failure, preemption, resource contention, etc. The handler is not called
 *                    if the container crashes or exits. The Pod's termination grace period countdown begins before the PreStop
 *                    hook is executed. Regardless of the outcome of the handler, the container will eventually terminate within
 *                    the Pod's termination grace period (unless delayed by finalizers). Other management of the container blocks
 *                    until the hook completes or until the termination grace period is reached. More info:
 *                    https://kubernetes.io/docs/concepts/containers/container-lifecycle-hooks/#container-hooks
*/
case class Lifecycle(
  postStart: Option[io.k8s.api.core.v1.LifecycleHandler] = None,
  preStop: Option[io.k8s.api.core.v1.LifecycleHandler] = None
)

object Lifecycle {
  implicit lazy val encoder: Encoder.AsObject[Lifecycle] = deriveEncoder
  implicit lazy val decoder: Decoder[Lifecycle] = deriveDecoder
}
