package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** NodeSystemInfo is a set of ids/uuids to uniquely identify the node.
 *
 *  @param               machineID  MachineID reported by the node. For unique machine identification in the cluster this field
 *                                  is preferred. Learn more from man(5) machine-id: http://man7.org/linux/man-pages/man5/machine-id.5.html
 *  @param              systemUUID  SystemUUID reported by the node. For unique machine identification MachineID is preferred.
 *                                  This field is specific to Red Hat hosts https://access.redhat.com/documentation/en-us/red_hat_subscription_management/1/html/rhsm/uuid
 *  @param                  bootID  Boot ID reported by the node.
 *  @param           kernelVersion  Kernel Version reported by the node from 'uname -r' (e.g. 3.16.0-0.bpo.4-amd64).
 *  @param                 osImage  OS Image reported by the node from /etc/os-release (e.g. Debian GNU/Linux 7 (wheezy)).
 *  @param containerRuntimeVersion  ContainerRuntime Version reported by the node through runtime remote API (e.g.
 *                                  containerd://1.4.2).
 *  @param          kubeletVersion  Kubelet Version reported by the node.
 *  @param        kubeProxyVersion  KubeProxy Version reported by the node.
 *  @param         operatingSystem  The Operating System reported by the node
 *  @param            architecture  The Architecture reported by the node
*/
case class NodeSystemInfo(
  machineID: String,
  systemUUID: String,
  bootID: String,
  kernelVersion: String,
  osImage: String,
  containerRuntimeVersion: String,
  kubeletVersion: String,
  kubeProxyVersion: String,
  operatingSystem: String,
  architecture: String
)

object NodeSystemInfo {
  implicit lazy val encoder: Encoder.AsObject[NodeSystemInfo] = deriveEncoder
  implicit lazy val decoder: Decoder[NodeSystemInfo] = deriveDecoder
}
