package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** Represents a Quobyte mount that lasts the lifetime of a pod. Quobyte volumes do not support ownership management or SELinux relabeling.
 *
 *  @param registry  registry represents a single or multiple Quobyte Registry services specified as a string as host:port
 *                   pair (multiple entries are separated with commas) which acts as the central registry for volumes
 *  @param   volume  volume is a string that references an already created Quobyte volume by name.
 *  @param    group  group to map volume access to Default is no group
 *  @param readOnly  readOnly here will force the Quobyte volume to be mounted with read-only permissions. Defaults to false
 *  @param   tenant  tenant owning the given Quobyte volume in the Backend Used with dynamically provisioned Quobyte volumes,
 *                   value is set by the plugin
 *  @param     user  user to map volume access to Defaults to serivceaccount user
*/
case class QuobyteVolumeSource(
  registry: String,
  volume: String,
  group: Option[String] = None,
  readOnly: Option[Boolean] = None,
  tenant: Option[String] = None,
  user: Option[String] = None
)

object QuobyteVolumeSource {
  implicit lazy val encoder: Encoder.AsObject[QuobyteVolumeSource] = deriveEncoder
  implicit lazy val decoder: Decoder[QuobyteVolumeSource] = deriveDecoder
}
