package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** ContainerStatus contains details for the current status of this container.
 *
 *  @param         name  This must be a DNS_LABEL. Each container in a pod must have a unique name. Cannot be updated.
 *  @param        ready  Specifies whether the container has passed its readiness probe.
 *  @param restartCount  The number of times the container has been restarted.
 *  @param        image  The image the container is running. More info: https://kubernetes.io/docs/concepts/containers/images
 *  @param      imageID  ImageID of the container's image.
 *  @param  containerID  Container's ID in the format '<type>://<container_id>'.
 *  @param    lastState  Details about the container's last termination condition.
 *  @param      started  Specifies whether the container has passed its startup probe. Initialized as false, becomes true after
 *                       startupProbe is considered successful. Resets to false when the container is restarted, or if kubelet
 *                       loses state temporarily. Is always true when no startupProbe is defined.
 *  @param        state  Details about the container's current condition.
*/
case class ContainerStatus(
  name: String,
  ready: Boolean,
  restartCount: Int,
  image: String,
  imageID: String,
  containerID: Option[String] = None,
  lastState: Option[io.k8s.api.core.v1.ContainerState] = None,
  started: Option[Boolean] = None,
  state: Option[io.k8s.api.core.v1.ContainerState] = None
)

object ContainerStatus {
  implicit lazy val encoder: Encoder.AsObject[ContainerStatus] = deriveEncoder
  implicit lazy val decoder: Decoder[ContainerStatus] = deriveDecoder
}
