package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** ISCSIPersistentVolumeSource represents an ISCSI disk. ISCSI volumes can only be mounted as read/write once. ISCSI volumes support ownership management and SELinux relabeling.
 *
 *  @param      targetPortal  targetPortal is iSCSI Target Portal. The Portal is either an IP or ip_addr:port if the port is
 *                            other than default (typically TCP ports 860 and 3260).
 *  @param               iqn  iqn is Target iSCSI Qualified Name.
 *  @param               lun  lun is iSCSI Target Lun number.
 *  @param chapAuthDiscovery  chapAuthDiscovery defines whether support iSCSI Discovery CHAP authentication
 *  @param   chapAuthSession  chapAuthSession defines whether support iSCSI Session CHAP authentication
 *  @param            fsType  fsType is the filesystem type of the volume that you want to mount. Tip: Ensure that the filesystem
 *                            type is supported by the host operating system. Examples: "ext4", "xfs", "ntfs". Implicitly inferred
 *                            to be "ext4" if unspecified. More info: https://kubernetes.io/docs/concepts/storage/volumes#iscsi
 *  @param     initiatorName  initiatorName is the custom iSCSI Initiator Name. If initiatorName is specified with iscsiInterface
 *                            simultaneously, new iSCSI interface <target portal>:<volume name> will be created for the connection
 *  @param    iscsiInterface  iscsiInterface is the interface Name that uses an iSCSI transport. Defaults to 'default' (tcp)
 *  @param           portals  portals is the iSCSI Target Portal List. The Portal is either an IP or ip_addr:port if the port
 *                            is other than default (typically TCP ports 860 and 3260).
 *  @param          readOnly  readOnly here will force the ReadOnly setting in VolumeMounts. Defaults to false.
 *  @param         secretRef  secretRef is the CHAP Secret for iSCSI target and initiator authentication
*/
case class ISCSIPersistentVolumeSource(
  targetPortal: String,
  iqn: String,
  lun: Int,
  chapAuthDiscovery: Option[Boolean] = None,
  chapAuthSession: Option[Boolean] = None,
  fsType: Option[String] = None,
  initiatorName: Option[String] = None,
  iscsiInterface: Option[String] = None,
  portals: Option[Seq[String]] = None,
  readOnly: Option[Boolean] = None,
  secretRef: Option[io.k8s.api.core.v1.SecretReference] = None
)

object ISCSIPersistentVolumeSource {
  implicit lazy val encoder: Encoder.AsObject[ISCSIPersistentVolumeSource] = deriveEncoder
  implicit lazy val decoder: Decoder[ISCSIPersistentVolumeSource] = deriveDecoder
}
