package io.k8s.api.apps.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** StatefulSetStatus represents the current state of a StatefulSet.
 *
 *  @param           replicas  replicas is the number of Pods created by the StatefulSet controller.
 *  @param  availableReplicas  Total number of available pods (ready for at least minReadySeconds) targeted by this statefulset
 *  @param     collisionCount  collisionCount is the count of hash collisions for the StatefulSet. The StatefulSet controller
 *                             uses this field as a collision avoidance mechanism when it needs to create the name for the newest
 *                             ControllerRevision.
 *  @param         conditions  Represents the latest available observations of a statefulset's current state.
 *  @param    currentReplicas  currentReplicas is the number of Pods created by the StatefulSet controller from the StatefulSet
 *                             version indicated by currentRevision.
 *  @param    currentRevision  currentRevision, if not empty, indicates the version of the StatefulSet used to generate Pods
 *                             in the sequence [0,currentReplicas).
 *  @param observedGeneration  observedGeneration is the most recent generation observed for this StatefulSet. It corresponds
 *                             to the StatefulSet's generation, which is updated on mutation by the API Server.
 *  @param      readyReplicas  readyReplicas is the number of pods created for this StatefulSet with a Ready Condition.
 *  @param     updateRevision  updateRevision, if not empty, indicates the version of the StatefulSet used to generate Pods
 *                             in the sequence [replicas-updatedReplicas,replicas)
 *  @param    updatedReplicas  updatedReplicas is the number of Pods created by the StatefulSet controller from the StatefulSet
 *                             version indicated by updateRevision.
*/
case class StatefulSetStatus(
  replicas: Int,
  availableReplicas: Option[Int] = None,
  collisionCount: Option[Int] = None,
  conditions: Option[Seq[io.k8s.api.apps.v1.StatefulSetCondition]] = None,
  currentReplicas: Option[Int] = None,
  currentRevision: Option[String] = None,
  observedGeneration: Option[Long] = None,
  readyReplicas: Option[Int] = None,
  updateRevision: Option[String] = None,
  updatedReplicas: Option[Int] = None
)

object StatefulSetStatus {
  implicit lazy val encoder: Encoder.AsObject[StatefulSetStatus] = deriveEncoder
  implicit lazy val decoder: Decoder[StatefulSetStatus] = deriveDecoder
}
