package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** ContainerPort represents a network port in a single container.
 *
 *  @param containerPort  Number of port to expose on the pod's IP address. This must be a valid port number, 0 < x < 65536.
 *  @param        hostIP  What host IP to bind the external port to.
 *  @param      hostPort  Number of port to expose on the host. If specified, this must be a valid port number, 0 < x < 65536.
 *                        If HostNetwork is specified, this must match ContainerPort. Most containers do not need this.
 *  @param          name  If specified, this must be an IANA_SVC_NAME and unique within the pod. Each named port in a pod must
 *                        have a unique name. Name for the port that can be referred to by services.
 *  @param      protocol  Protocol for port. Must be UDP, TCP, or SCTP. Defaults to "TCP".
*/
case class ContainerPort(
  containerPort: Int,
  hostIP: Option[String] = None,
  hostPort: Option[Int] = None,
  name: Option[String] = None,
  protocol: Option[String] = None
)

object ContainerPort {
  implicit lazy val encoder: Encoder.AsObject[ContainerPort] = deriveEncoder
  implicit lazy val decoder: Decoder[ContainerPort] = deriveDecoder
}
