package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** ContainerStatus contains details for the current status of this container.
 *
 *  @param                     name  Name is a DNS_LABEL representing the unique name of the container. Each container in a
 *                                   pod must have a unique name across all container types. Cannot be updated.
 *  @param                    ready  Ready specifies whether the container is currently passing its readiness check. The value
 *                                   will change as readiness probes keep executing. If no readiness probes are specified, this
 *                                   field defaults to true once the container is fully started (see Started field).
 *                                   The value is typically used to determine whether a container is ready to accept traffic.
 *  @param             restartCount  RestartCount holds the number of times the container has been restarted. Kubelet makes
 *                                   an effort to always increment the value, but there are cases when the state may be lost
 *                                   due to node restarts and then the value may be reset to 0. The value is never negative.
 *  @param                    image  Image is the name of container image that the container is running. The container image
 *                                   may not match the image used in the PodSpec, as it may have been resolved by the runtime.
 *                                   More info: https://kubernetes.io/docs/concepts/containers/images.
 *  @param                  imageID  ImageID is the image ID of the container's image. The image ID may not match the image
 *                                   ID of the image used in the PodSpec, as it may have been resolved by the runtime.
 *  @param       allocatedResources  AllocatedResources represents the compute resources allocated for this container by the
 *                                   node. Kubelet sets this value to Container.Resources.Requests upon successful pod admission
 *                                   and after successfully admitting desired pod resize.
 *  @param allocatedResourcesStatus  AllocatedResourcesStatus represents the status of various resources allocated for this
 *                                   Pod.
 *  @param              containerID  ContainerID is the ID of the container in the format '<type>://<container_id>'. Where
 *                                   type is a container runtime identifier, returned from Version call of CRI API (for example
 *                                   "containerd").
 *  @param                lastState  LastTerminationState holds the last termination state of the container to help debug container
 *                                   crashes and restarts. This field is not populated if the container is still running and
 *                                   RestartCount is 0.
 *  @param                resources  Resources represents the compute resource requests and limits that have been successfully
 *                                   enacted on the running container after it has been started or has been successfully resized
 *  @param                  started  Started indicates whether the container has finished its postStart lifecycle hook and
 *                                   passed its startup probe. Initialized as false, becomes true after startupProbe is considered
 *                                   successful. Resets to false when the container is restarted, or if kubelet loses state
 *                                   temporarily. In both cases, startup probes will run again. Is always true when no startupProbe
 *                                   is defined and container is running and has passed the postStart lifecycle hook. The null
 *                                   value must be treated the same as false.
 *  @param                    state  State holds details about the container's current condition.
 *  @param                     user  User represents user identity information initially attached to the first process of the
 *                                   container
 *  @param             volumeMounts  Status of volume mounts.
*/
case class ContainerStatus(
  name: String,
  ready: Boolean,
  restartCount: Int,
  image: String,
  imageID: String,
  allocatedResources: Option[Map[String, io.k8s.apimachinery.pkg.api.resource.Quantity]] = None,
  allocatedResourcesStatus: Option[Seq[io.k8s.api.core.v1.ResourceStatus]] = None,
  containerID: Option[String] = None,
  lastState: Option[io.k8s.api.core.v1.ContainerState] = None,
  resources: Option[io.k8s.api.core.v1.ResourceRequirements] = None,
  started: Option[Boolean] = None,
  state: Option[io.k8s.api.core.v1.ContainerState] = None,
  user: Option[io.k8s.api.core.v1.ContainerUser] = None,
  volumeMounts: Option[Seq[io.k8s.api.core.v1.VolumeMountStatus]] = None
)

object ContainerStatus {
  implicit lazy val encoder: Encoder.AsObject[ContainerStatus] = deriveEncoder
  implicit lazy val decoder: Decoder[ContainerStatus] = deriveDecoder
}
