package io.k8s.api.apps.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** DaemonSetStatus represents the current status of a daemon set.
 *
 *  @param currentNumberScheduled  The number of nodes that are running at least 1 daemon pod and are supposed to run the daemon
 *                                 pod. More info: https://kubernetes.io/docs/concepts/workloads/controllers/daemonset/
 *  @param     numberMisscheduled  The number of nodes that are running the daemon pod, but are not supposed to run the daemon
 *                                 pod. More info: https://kubernetes.io/docs/concepts/workloads/controllers/daemonset/
 *  @param desiredNumberScheduled  The total number of nodes that should be running the daemon pod (including nodes correctly
 *                                 running the daemon pod). More info: https://kubernetes.io/docs/concepts/workloads/controllers/daemonset/
 *  @param            numberReady  numberReady is the number of nodes that should be running the daemon pod and have one or
 *                                 more of the daemon pod running with a Ready Condition.
 *  @param         collisionCount  Count of hash collisions for the DaemonSet. The DaemonSet controller uses this field as
 *                                 a collision avoidance mechanism when it needs to create the name for the newest ControllerRevision
 *  @param             conditions  Represents the latest available observations of a DaemonSet's current state.
 *  @param        numberAvailable  The number of nodes that should be running the daemon pod and have one or more of the daemon
 *                                 pod running and available (ready for at least spec.minReadySeconds)
 *  @param      numberUnavailable  The number of nodes that should be running the daemon pod and have none of the daemon pod
 *                                 running and available (ready for at least spec.minReadySeconds)
 *  @param     observedGeneration  The most recent generation observed by the daemon set controller.
 *  @param updatedNumberScheduled  The total number of nodes that are running updated daemon pod
*/
case class DaemonSetStatus(
  currentNumberScheduled: Int,
  numberMisscheduled: Int,
  desiredNumberScheduled: Int,
  numberReady: Int,
  collisionCount: Option[Int] = None,
  conditions: Option[Seq[io.k8s.api.apps.v1.DaemonSetCondition]] = None,
  numberAvailable: Option[Int] = None,
  numberUnavailable: Option[Int] = None,
  observedGeneration: Option[Long] = None,
  updatedNumberScheduled: Option[Int] = None
)

object DaemonSetStatus {
  implicit lazy val encoder: Encoder.AsObject[DaemonSetStatus] = deriveEncoder
  implicit lazy val decoder: Decoder[DaemonSetStatus] = deriveDecoder
}
