package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** ObjectReference contains enough information to let you inspect or modify the referred object.
 *
 *  @param      apiVersion  API version of the referent.
 *  @param       fieldPath  If referring to a piece of an object instead of an entire object, this string should contain a
 *                          valid JSON/Go field access statement, such as desiredState.manifest.containers[2]. For example,
 *                          if the object reference is to a container within a pod, this would take on a value like: "spec.containers{name}"
 *                          (where "name" refers to the name of the container that triggered the event) or if no container name
 *                          is specified "spec.containers[2]" (container with index 2 in this pod). This syntax is chosen only
 *                          to have some well-defined way of referencing a part of an object.
 *  @param            kind  Kind of the referent. More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#types-kinds
 *  @param            name  Name of the referent. More info: https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#names
 *  @param       namespace  Namespace of the referent. More info: https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/
 *  @param resourceVersion  Specific resourceVersion to which this reference is made, if any. More info:
 *                          https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#concurrency-control-and-consistency
 *  @param             uid  UID of the referent. More info: https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#uids
*/
case class ObjectReference(
  apiVersion: Option[String] = None,
  fieldPath: Option[String] = None,
  kind: Option[String] = None,
  name: Option[String] = None,
  namespace: Option[String] = None,
  resourceVersion: Option[String] = None,
  uid: Option[String] = None
)

object ObjectReference {
  implicit lazy val encoder: Encoder.AsObject[ObjectReference] = deriveEncoder
  implicit lazy val decoder: Decoder[ObjectReference] = deriveDecoder
}
