package io.k8s.api.core.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** ScaleIOPersistentVolumeSource represents a persistent ScaleIO volume
 *
 *  @param          gateway  gateway is the host address of the ScaleIO API Gateway.
 *  @param           system  system is the name of the storage system as configured in ScaleIO.
 *  @param        secretRef  secretRef references to the secret for ScaleIO user and other sensitive information. If this is
 *                           not provided, Login operation will fail.
 *  @param           fsType  fsType is the filesystem type to mount. Must be a filesystem type supported by the host operating
 *                           system. Ex. "ext4", "xfs", "ntfs". Default is "xfs"
 *  @param protectionDomain  protectionDomain is the name of the ScaleIO Protection Domain for the configured storage.
 *  @param         readOnly  readOnly defaults to false (read/write). ReadOnly here will force the ReadOnly setting in VolumeMounts
 *  @param       sslEnabled  sslEnabled is the flag to enable/disable SSL communication with Gateway, default false
 *  @param      storageMode  storageMode indicates whether the storage for a volume should be ThickProvisioned or ThinProvisioned.
 *                           Default is ThinProvisioned.
 *  @param      storagePool  storagePool is the ScaleIO Storage Pool associated with the protection domain.
 *  @param       volumeName  volumeName is the name of a volume already created in the ScaleIO system that is associated with
 *                           this volume source.
*/
case class ScaleIOPersistentVolumeSource(
  gateway: String,
  system: String,
  secretRef: io.k8s.api.core.v1.SecretReference,
  fsType: Option[String] = None,
  protectionDomain: Option[String] = None,
  readOnly: Option[Boolean] = None,
  sslEnabled: Option[Boolean] = None,
  storageMode: Option[String] = None,
  storagePool: Option[String] = None,
  volumeName: Option[String] = None
)

object ScaleIOPersistentVolumeSource {
  implicit lazy val encoder: Encoder.AsObject[ScaleIOPersistentVolumeSource] = deriveEncoder
  implicit lazy val decoder: Decoder[ScaleIOPersistentVolumeSource] = deriveDecoder
}
