package io.k8s.api.storage.v1

import io.circe.*
import io.circe.generic.semiauto.*

/** StorageClass describes the parameters for a class of storage for which PersistentVolumes can be dynamically provisioned.

StorageClasses are non-namespaced; the name of the storage class according to etcd is in ObjectMeta.Name.
 *
 *  @param          provisioner  provisioner indicates the type of the provisioner.
 *  @param allowVolumeExpansion  allowVolumeExpansion shows whether the storage class allow volume expand.
 *  @param    allowedTopologies  allowedTopologies restrict the node topologies where volumes can be dynamically provisioned.
 *                               Each volume plugin defines its own supported topology specifications. An empty TopologySelectorTerm
 *                               list means there is no topology restriction. This field is only honored by servers that enable
 *                               the VolumeScheduling feature.
 *  @param           apiVersion  APIVersion defines the versioned schema of this representation of an object. Servers should
 *                               convert recognized schemas to the latest internal value, and may reject unrecognized values.
 *                               More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#resources
 *  @param                 kind  Kind is a string value representing the REST resource this object represents. Servers may
 *                               infer this from the endpoint the client submits requests to. Cannot be updated. In CamelCase.
 *                               More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#types-kinds
 *  @param             metadata  Standard object's metadata. More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#metadata
 *  @param         mountOptions  mountOptions controls the mountOptions for dynamically provisioned PersistentVolumes of this
 *                               storage class. e.g. ["ro", "soft"]. Not validated - mount of the PVs will simply fail if one
 *                               is invalid.
 *  @param           parameters  parameters holds the parameters for the provisioner that should create volumes of this storage
 *                               class.
 *  @param        reclaimPolicy  reclaimPolicy controls the reclaimPolicy for dynamically provisioned PersistentVolumes of
 *                               this storage class. Defaults to Delete.
 *  @param    volumeBindingMode  volumeBindingMode indicates how PersistentVolumeClaims should be provisioned and bound.  When
 *                               unset, VolumeBindingImmediate is used. This field is only honored by servers that enable the
 *                               VolumeScheduling feature.
*/
case class StorageClass(
  provisioner: String,
  allowVolumeExpansion: Option[Boolean] = None,
  allowedTopologies: Option[Seq[io.k8s.api.core.v1.TopologySelectorTerm]] = None,
  apiVersion: Option[String] = None,
  kind: Option[String] = None,
  metadata: Option[io.k8s.apimachinery.pkg.apis.meta.v1.ObjectMeta] = None,
  mountOptions: Option[Seq[String]] = None,
  parameters: Option[Map[String, String]] = None,
  reclaimPolicy: Option[String] = None,
  volumeBindingMode: Option[String] = None
)

object StorageClass {
  implicit lazy val encoder: Encoder.AsObject[StorageClass] = deriveEncoder
  implicit lazy val decoder: Decoder[StorageClass] = deriveDecoder
}
