package com.gradle.develocity.agent.maven.api.cache;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.project.MavenProject;

import java.util.List;
import java.util.function.Consumer;

/**
 * Allows users to configure normalization.
 *
 * @since 1.21
 */
public interface NormalizationProvider {

    /**
     * Configure normalization for the project in the supplied context.
     * <p>
     * This method is called once for each project.
     *
     * @param context provides access to the project and allows users to customize its runtime classpath normalization
     */
    void configureNormalization(Context context);

    /**
     * Allows configuring the normalization of a project.
     *
     * @since 1.21
     */
    interface Context {

        /**
         * The project being configured.
         */
        MavenProject getProject();

        /**
         * The session associated with the project.
         */
        MavenSession getSession();

        /**
         * Allows configuring the runtime classpath normalization.
         *
         * @param action an action to configure the runtime classpath normalization
         */
        Context configureRuntimeClasspathNormalization(Consumer<RuntimeClasspathNormalization> action);

        /**
         * Allows configuring the system property normalization.
         *
         * @param action an action to configure the system property normalization
         */
        Context configureSystemPropertiesNormalization(Consumer<SystemPropertiesNormalization> action);
    }

    /**
     * Allows configuring the runtime classpath normalization.
     * <p>
     * Supports ANT-style patterns:
     * <pre>
     *     * matches zero or more characters
     *     ? matches one character
     *     ** matches zero or more characters across directory levels
     * </pre>
     *
     * @since 1.21
     */
    interface RuntimeClasspathNormalization {

        /**
         * Configures the runtime classpath normalization to ignore files. Supports ANT-style file paths.
         * This operation will override any previous configuration.
         *
         * @param ignoredFiles the files to ignore
         */
        RuntimeClasspathNormalization setIgnoredFiles(List<String> ignoredFiles);

        /**
         * Configures the runtime classpath normalization to ignore files. Supports ANT-style file paths.
         * This operation will override any previous configuration.
         *
         * @param ignoredFiles the files to ignore
         */
        RuntimeClasspathNormalization setIgnoredFiles(String... ignoredFiles);

        /**
         * Configures the runtime classpath normalization to ignore files. Supports ANT-style file paths.
         * This operation is additive, i.e. it will add the ignored files to the previously configured files
         * including the corresponding configuration in POM.
         *
         * @param ignoredFiles the files to ignore
         */
        RuntimeClasspathNormalization addIgnoredFiles(List<String> ignoredFiles);

        /**
         * Configures the runtime classpath normalization to ignore files. Supports ANT-style file paths.
         * This operation is additive, i.e. it will add the ignored files to the previously configured files.
         *
         * @param ignoredFiles the files to ignore
         */
        RuntimeClasspathNormalization addIgnoredFiles(String... ignoredFiles);

        /**
         * Configures the runtime classpath normalization to ignore properties from specific property files.
         * ANT-style patterns can be used to match relevant property files.
         *
         * @param path              the path of the properties file
         * @param ignoredProperties the properties to ignore
         */
        RuntimeClasspathNormalization addPropertiesNormalization(String path, List<String> ignoredProperties);

        /**
         * Configures the runtime classpath normalization to ignore properties from specific property files.
         * ANT-style patterns can be used to match relevant property files.
         *
         * @param path              the path of the properties file
         * @param ignoredProperties the properties to ignore
         */
        RuntimeClasspathNormalization addPropertiesNormalization(String path, String... ignoredProperties);

        /**
         * Allows configuring the normalization of files matching <code>META-INF/**&#47;*.properties</code>.
         *
         * @param action an action to configure the normalization
         */
        RuntimeClasspathNormalization configureMetaInf(Consumer<RuntimeClasspathNormalization.MetaInf> action);

        /**
         * A common location to store properties files is the {@code META-INF} directory.
         * Thus, this provides a shortcut for normalizing files matching <code>META-INF/**&#47;*.properties</code>.
         *
         * @since 1.21
         */
        interface MetaInf {

            /**
             * Allows ignoring one or more attributes in {@code MANIFEST} files. This operation will override any previous configuration.
             *
             * @param ignoredAttributes the attributes to ignore
             */
            RuntimeClasspathNormalization.MetaInf setIgnoredAttributes(List<String> ignoredAttributes);

            /**
             * Allows ignoring one or more attributes in {@code MANIFEST} files. This operation will override any previous configuration.
             *
             * @param ignoredAttributes the attributes to ignore
             */
            RuntimeClasspathNormalization.MetaInf setIgnoredAttributes(String... ignoredAttributes);

            /**
             * Allows ignoring one or more attributes in {@code MANIFEST} files.
             * This operation is additive, i.e. it will add the ignored attributes to the previously configured attributes.
             *
             * @param ignoredAttributes the attributes to ignore
             */
            RuntimeClasspathNormalization.MetaInf addIgnoredAttributes(List<String> ignoredAttributes);

            /**
             * Allows ignoring one or more attributes in {@code MANIFEST} files.
             * This operation is additive, i.e. it will add the ignored attributes to the previously configured attributes.
             *
             * @param ignoredAttributes the attributes to ignore
             */
            RuntimeClasspathNormalization.MetaInf addIgnoredAttributes(String... ignoredAttributes);

            /**
             * Allows ignoring one or more properties. This operation will override any previous configuration.
             *
             * @param ignoredProperties the properties to ignore
             */
            RuntimeClasspathNormalization.MetaInf setIgnoredProperties(List<String> ignoredProperties);

            /**
             * Allows ignoring one or more properties. This operation will override any previous configuration.
             *
             * @param ignoredProperties the properties to ignore
             */
            RuntimeClasspathNormalization.MetaInf setIgnoredProperties(String... ignoredProperties);

            /**
             * Allows ignoring one or more properties. This operation is additive,
             * i.e. it will add the ignored properties to the previously configured properties.
             *
             * @param ignoredProperties the properties to ignore
             */
            RuntimeClasspathNormalization.MetaInf addIgnoredProperties(List<String> ignoredProperties);

            /**
             * Allows ignoring one or more properties. This operation is additive,
             * i.e. it will add the ignored properties to the previously configured properties.
             *
             * @param ignoredProperties the properties to ignore
             */
            RuntimeClasspathNormalization.MetaInf addIgnoredProperties(String... ignoredProperties);

            /**
             * Allows ignoring the {@code MANIFEST} files completely.
             *
             * @param ignoreManifest whether to ignore the {@code MANIFEST} files
             */
            RuntimeClasspathNormalization.MetaInf setIgnoreManifest(boolean ignoreManifest);

            /**
             * Allows ignoring all files in the {@code META-INF} directories.
             *
             * @param ignoreCompletely whether to completely ignore the files in {@code META-INF} directories
             */
            RuntimeClasspathNormalization.MetaInf setIgnoreCompletely(boolean ignoreCompletely);
        }
    }

    /**
     * Allows configuring the system property normalization of a project.
     *
     * @since 1.21
     */
    interface SystemPropertiesNormalization {

        /**
         * Allows configuring the normalization to ignore system properties supplied.
         *
         * @param systemPropertyNames the system properties to be ignored
         */
        SystemPropertiesNormalization setIgnoredKeys(String... systemPropertyNames);

        /**
         * Allows configuring the normalization to ignore system properties supplied.
         *
         * @param systemPropertyNames the system properties to be ignored
         */
        SystemPropertiesNormalization setIgnoredKeys(List<String> systemPropertyNames);

        /**
         * Allows configuring the normalization to ignore system properties supplied.
         *
         * @param systemPropertyNames the system properties to be ignored
         */
        SystemPropertiesNormalization addIgnoredKeys(String... systemPropertyNames);

        /**
         * Allows configuring the normalization to ignore system properties supplied.
         *
         * @param systemPropertyNames the system properties to be ignored
         */
        SystemPropertiesNormalization addIgnoredKeys(List<String> systemPropertyNames);
    }
}
