package com.gradle.develocity.agent.maven.api.scan;

import javax.annotation.Nullable;
import java.net.URI;
import java.util.function.Consumer;

/**
 * Allows to interact with the build scan feature of the Develocity Maven extension.
 *
 * @since 1.21
 */
public interface BuildScanApi {

    /**
     * Captures a tag for the current build. The tag is not required to be unique for a given build.
     *
     * @param tag the name of the tag, must not be {@code null}
     */
    void tag(String tag);

    /**
     * Captures a named value for the current build. The name is not required to be unique for a given build.
     *
     * @param name  the name, must not be {@code null}
     * @param value the value or {@code null}
     */
    void value(String name, @Nullable String value);

    /**
     * Captures a named link for the current build. The name is not required to be unique for a given build.
     *
     * @param name the name, must not be {@code null}
     * @param url  the url, must not be {@code null}, must be a valid http:, https: or mailto: URL
     */
    void link(String name, String url);

    /**
     * Executes the given action in a background thread, allowing the current Maven work to continue.
     * <p>
     * This method is useful for capturing values, tags and links that are expensive to compute.
     * By capturing them in the background, Maven can continue doing other work, making your build faster.
     * <p>
     * For example, if you are capturing the Git commit ID as a custom value you should invoke Git in the background:
     * <pre>
     * import org.eclipse.jgit.*;
     * background(api -&gt; {
     *   Git git = ...
     *   ObjectId objectId = git.getRepository().resolve("HEAD");
     *   api.value("Git Commit ID", ObjectId.toString(objectId));
     * });
     * </pre>
     * <p>
     * All background work will be completed before finishing the build and publishing the build scan.
     * <p>
     * Any errors that are thrown by the background action will be logged and captured in the build scan.
     *
     * @param action the action to execute in the background
     */
    void background(Consumer<? super com.gradle.develocity.agent.maven.api.scan.BuildScanApi> action);

    /**
     * Registers a callback that is invoked when the build has finished, but <i>before</i> this extension stops
     * accepting to be called.
     *
     * @param action the action to execute when the build has finished
     */
    void buildFinished(Consumer<? super BuildResult> action);

    /**
     * Registers a callback that is invoked when a build scan has been published successfully.
     * <p>
     * If {@link #isUploadInBackground()} is {@code true}, the callback is invoked as early as possible,
     * which is as soon as the Develocity server provides a build scan URL, and <i>before</i> the actual upload to the server.
     * This means that if the URL is accessed right away, it might not be available yet, as the build scan is still being transmitted and processed by the server.
     *
     * @param action the action to execute when the build scan has been published successfully
     */
    void buildScanPublished(Consumer<? super PublishedBuildScan> action);

    /**
     * The location of the Gradle Terms of Use that are agreed to when creating a build scan.
     * <p>
     * Configuration via the {@code develocity.scan.termsOfUse.url} system property will always take precedence.
     *
     * @param termsOfUseUrl the location of the Gradle Terms of Use
     */
    void setTermsOfUseUrl(String termsOfUseUrl);

    /**
     * The location of the Gradle Terms of Use that are agreed to when creating a build scan.
     *
     * @return the location of the Gradle Terms of Use or {@code null}
     */
    @Nullable
    String getTermsOfUseUrl();

    /**
     * Indicates whether the Gradle Terms of Use specified under {@link #setTermsOfUseUrl(String)} are agreed to.
     * <p>
     * Configuration via the {@code develocity.scan.termsOfUse.accept} system property will always take precedence.
     *
     * @param agree <i>true</i> if agreeing to the Gradle Terms of Use, false otherwise
     */
    void setTermsOfUseAgree(String agree);

    /**
     * The agreement of the Gradle Terms of Use specified under {@link #setTermsOfUseUrl(String)}.
     *
     * @return the agreement of the Gradle Terms of Use or {@code null}
     */
    @Nullable
    String getTermsOfUseAgree();

    /**
     * Sets the URL of the Develocity server to which the build scans are published.
     *
     * @param url the server URL or {@code null}
     */
    default void setServer(@Nullable String url) {
        setServer(url == null ? null : URI.create(url));
    }

    /**
     * Sets the URL of the Develocity server to which the build scans are published.
     *
     * @param url the server URL or {@code null}
     */
    void setServer(@Nullable URI url);

    /**
     * Returns the URL of the Develocity server to which the build scans are published.
     *
     * @return the Develocity server or {@code null}
     */
    @Nullable
    String getServer();

    /**
     * Specifies whether it is acceptable to communicate with a Develocity server using an untrusted SSL certificate.
     * <p>
     * The default (public) build scan server uses SSL certificates that are trusted by default by standard modern Java environments.
     * If you are using a different build scan server via Develocity, it may use an untrusted certificate.
     * This may be due to the use of an internally provisioned or self-signed certificate.
     * <p>
     * In such a scenario, you can either configure the build JVM environment to trust the certificate,
     * or call this method with {@code true} to disable verification of the server's identity.
     * Alternatively, you may disable SSL completely for Develocity installation but this is not recommended.
     * <p>
     * Allowing communication with untrusted servers keeps data encrypted during transmission,
     * but makes it easy for a man-in-the-middle to impersonate the intended server and capture data.
     * <p>
     * This value has no effect if a server is specified using the HTTP protocol (i.e. has SSL disabled).
     *
     * @param allow whether to allow communication with an HTTPS server with an untrusted certificate
     */
    void setAllowUntrustedServer(boolean allow);

    /**
     * Whether it is acceptable to communicate with a build scan server with an untrusted SSL certificate.
     *
     * @return {@code true} it is acceptable to communicate with a build scan server with an untrusted SSL certificate
     */
    boolean getAllowUntrustedServer();

    /**
     * Specifies whether to upload the build scan in background after the build has finished.
     * <p>
     * Defaults to {@code true}.
     * <p>
     * This allows the build to finish sooner, but can be problematic in build environments that
     * terminate as soon as the build is finished as the upload may be terminated before it completes.
     * Background uploading should be disabled for such environments.
     * <p>
     * This property may also be set in {@code develocity.xml} or via the {@code develocity.scan.uploadInBackground} system property,
     * which if set takes precedence over any value set by this method and the default.
     * If this is set to any value other than {@code true}, the background build scan upload will be disabled.
     * <p>
     * This method cannot be called after the MavenSession has been started (i.e. after the {@link org.apache.maven.execution.ExecutionEvent.Type#SessionStarted} event has been received).
     * Doing so will produce a build time error.
     *
     * @param uploadInBackground whether to upload the build scan in background
     */
    void setUploadInBackground(boolean uploadInBackground);

    /**
     * See {@link #setUploadInBackground(boolean)}.
     *
     * @return whether to upload build scan in background
     */
    boolean isUploadInBackground();

    /**
     * Executes the given action only once over the course of the Maven build execution.
     * <p>
     * The identifier is used to uniquely identify a given action.
     * Calling this API a second time with the same identifier will not run the action again.
     * <p>
     * Any errors that are thrown by the action will be logged and captured in the build scan. In case of a failure while executing the action for a given identifier, no other action will be executed for the same identifier.
     *
     * @param identifier a unique identifier
     * @param action     the action to execute only once
     */
    void executeOnce(String identifier, Consumer<? super com.gradle.develocity.agent.maven.api.scan.BuildScanApi> action);

    /**
     * Allows registering functions for obfuscating certain identifying information within build scans.
     *
     * @return the register of obfuscation functions
     */
    BuildScanDataObfuscation getObfuscation();

    /**
     * Allows registering functions for obfuscating certain identifying information within build scans.
     *
     * @param action a function to be applied to the register of obfuscation functions
     */
    void obfuscation(Consumer<? super BuildScanDataObfuscation> action);

    /**
     * Allows configuring what data will be captured as part of the build scan.
     *
     * @return the capture settings
     */
    BuildScanCaptureSettings getCapture();

    /**
     * Allows configuring what data will be captured as part of the build scan.
     *
     * @param action a function to be applied to the capture settings
     */
    void capture(Consumer<? super BuildScanCaptureSettings> action);

    /**
     * Retrieves the build scan publication settings.
     *
     * @return the publishing settings
     */
    BuildScanPublishing getPublishing();

    /**
     * Allows configuring whether a build scan should be published at the end of the build.
     *
     * @param action a function to be applied to the publishing settings
     */
    void publishing(Consumer<? super BuildScanPublishing> action);
}
