package com.gradle.maven.extension.api.cache;

/**
 * Configures the local and remote build cache.
 *
 * @since 1.6.5
 */
public interface BuildCacheApi {

    /**
     * Gets the local build cache configuration.
     *
     * @return the local build cache configuration
     * @since 1.6.5
     */
    LocalBuildCache getLocal();

    /**
     * Gets the remote build cache configuration.
     *
     * @return the remote build cache configuration
     * @since 1.6.5
     */
    RemoteBuildCache getRemote();

    /**
     * Whether the {@code clean} lifecycle phase is required in order to use the build cache.
     *
     * @return {@code true} if {@code clean} is required; {@code false}, otherwise
     * @since 1.10.3
     */
    boolean isRequireClean();

    /**
     * Sets whether the {@code clean} lifecycle phase is required in order to use the build cache.
     * <p>
     * You should only set this to {@code false} when the build is started from a clean working directory.
     * Otherwise, produced build cache entries may be incorrect.
     * <p>
     * Configuration via the {@code gradle.cache.requireClean} system property will always take precedence.
     *
     * @param requireClean whether {@code clean} is required in order to use the build cache.
     * @since 1.10.3
     */
    void setRequireClean(boolean requireClean);

    /**
     * Registers a {@link MojoMetadataProvider}, which allows users to configure caching for Mojos
     * that the Gradle Enterprise Maven extension does not support out of the box.
     *
     * @param metadataProvider the metadata provider, must not be null
     * @since 1.15
     */
    void registerMojoMetadataProvider(MojoMetadataProvider metadataProvider);

    /**
     * Registers a {@link NormalizationProvider}, which allows to configure
     * runtime classpath normalization.
     * <p>
     * This can be used to ignore volatile data on the runtime classpath that does not
     * influence the outcome of goals, e.g. test goals. For more information, please consult
     * the <a href="https://docs.gradle.com/enterprise/maven-extension/#normalization">user manual</a>.
     *
     * @param normalizationProvider the normalization provider, must not be null
     * @since 1.15
     */
    void registerNormalizationProvider(NormalizationProvider normalizationProvider);
}
