package com.gradle.maven.extension.api;

import com.gradle.maven.extension.api.cache.BuildCacheApi;
import com.gradle.maven.extension.api.scan.BuildScanApi;
import com.gradle.nullability.Nullable;

import java.net.URI;
import java.nio.file.Path;

/**
 * Allows to interact with the Gradle Enterprise Maven extension.
 *
 * @since 1.10.1
 */
public interface GradleEnterpriseApi {

    /**
     * Whether the Gradle Enterprise Maven extension is enabled.
     *
     * @return {@code true} if the Gradle Enterprise Maven extension is enabled, {@code false} otherwise
     * @since 1.10.3
     */
    boolean isEnabled();

    /**
     * Sets whether to enable the Gradle Enterprise Maven extension.
     * <p>
     * Configuration via the {@code gradle.enterprise.enabled} system property will always take precedence.
     *
     * @param enabled whether to enable the Gradle Enterprise Maven extension
     * @since 1.10.3
     */
    void setEnabled(boolean enabled);

    /**
     * Sets the project identifier. The value must not be empty or exceed 256 characters.
     *
     * @param projectId the project identifier
     * @since 1.19
     */
    void setProjectId(@Nullable String projectId);

    /**
     * Returns the project identifier.
     *
     * @return null when no project identifier is configured
     * @since 1.19
     */
    @Nullable
    String getProjectId();

    /**
     * Returns the Gradle Enterprise Maven extension storage directory.
     *
     * @return the Gradle Enterprise Maven extension storage directory.
     * @see <a href="https://docs.gradle.com/enterprise/maven-extension/#anatomy_of_the_gradle_enterprise_directory">Gradle Enterprise Maven extension documentation</a>.
     * @since 1.10.1
     */
    Path getStorageDirectory();

    /**
     * Sets the Gradle Enterprise Maven extension storage directory to the specified path.
     * <p>
     * Configuration via the {@code gradle.enterprise.storage.directory} system property will always take precedence.
     *
     * @param path The new storage directory
     * @see <a href="https://docs.gradle.com/enterprise/maven-extension/#anatomy_of_the_gradle_enterprise_directory">Gradle Enterprise Maven extension documentation</a>.
     * @since 1.10.1
     */
    void setStorageDirectory(Path path);

    /**
     * Sets the URL of the Gradle Enterprise server.
     * <p>
     * Configuration via the {@code gradle.enterprise.url} system property will always take precedence.
     *
     * @param url the server URL
     * @since 1.10.1
     */
    default void setServer(String url) {
        setServer(url == null ? null : URI.create(url));
    }

    /**
     * Sets the URL of the Gradle Enterprise server.
     * <p>
     * Configuration via the {@code gradle.enterprise.url} system property will always take precedence.
     *
     * @param url the server URL
     * @since 1.10.3
     */
    void setServer(URI url);

    /**
     * Returns the URL of the Gradle Enterprise server.
     *
     * @return null when no Gradle Enterprise server is configured
     * @since 1.10.1
     */
    @Nullable
    String getServer();

    /**
     * Specifies whether it is acceptable to communicate with a Gradle Enterprise server using an untrusted SSL certificate.
     * <p>
     * The default (public) Gradle Enterprise server uses SSL certificates that are trusted by default by standard modern Java environments.
     * If you are using a different Gradle Enterprise server, it may use an untrusted certificate.
     * This may be due to the use of an internally provisioned or self-signed certificate.
     * <p>
     * In such a scenario, you can either configure the build JVM environment to trust the certificate,
     * or call this method with {@code true} to disable verification of the server's identity.
     * Alternatively, you may disable SSL completely for Gradle Enterprise installation but this is not recommended.
     * <p>
     * Allowing communication with untrusted servers keeps data encrypted during transmission,
     * but makes it easy for a man-in-the-middle to impersonate the intended server and capture data.
     * <p>
     * This value has no effect if a server is specified using the HTTP protocol (i.e. has SSL disabled).
     * <p>
     * Configuration via the {@code gradle.enterprise.allowUntrustedServer} system property will always take precedence.
     *
     * @param allow whether to allow communication with a HTTPS server with an untrusted certificate
     * @since 1.10.1
     */
    void setAllowUntrustedServer(boolean allow);

    /**
     * Whether it is acceptable to communicate with a Gradle Enterprise server with an untrusted SSL certificate.
     *
     * @return <code>true</code> it is acceptable to communicate with a build scan server with an untrusted SSL certificate
     * @since 1.10.1
     **/
    boolean getAllowUntrustedServer();

    /**
     * Sets the access key for authenticating with the Gradle Enterprise server.
     * <p>
     * An access key configured this way will take precedence over the {@code GRADLE_ENTERPRISE_ACCESS_KEY}
     * environment variable or access key file entry associated with the server.
     *
     * @param accessKey a Gradle Enterprise server access key without any hostname prefix
     * @since 1.11
     */
    void setAccessKey(String accessKey);

    /**
     * Returns the access key for authenticating with the Gradle Enterprise server.
     * <p>
     * Only the value of the explicitly configured access key (via {@link #setAccessKey(String)} or {@code gradle-enterprise.xml}) is returned but
     * not the value of an access key configured via the {@code GRADLE_ENTERPRISE_ACCESS_KEY} environment variable or access key file entry.
     *
     * @return the configured Gradle Enterprise server access key, if available; otherwise, {@code null}
     * @since 1.11
     */
    String getAccessKey();

    /**
     * The build scan API.
     *
     * @return the build scan API
     * @since 1.10.1
     */
    BuildScanApi getBuildScan();

    /**
     * The build cache API.
     *
     * @return the build cache API
     * @since 1.10.1
     */
    BuildCacheApi getBuildCache();

}
