/*
 * Decompiled with CFR 0.152.
 */
package com.gradle;

import com.gradle.maven.extension.api.cache.BuildCacheApi;
import com.gradle.maven.extension.api.cache.MojoMetadataProvider;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.project.MavenProject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class QuarkusBuildCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(QuarkusBuildCache.class);
    private static final List<String> QUARKUS_CONFIG_KEY_NATIVE_CONTAINER_BUILD = Arrays.asList("quarkus.native.container-build", "quarkus.native.remote-container-build");
    private static final String QUARKUS_CONFIG_KEY_NATIVE_BUILDER_IMAGE = "quarkus.native.builder-image";
    private static final String QUARKUS_CONFIG_KEY_PACKAGE_TYPE = "quarkus.package.type";
    private static final String QUARKUS_CONFIG_KEY_GRAALVM_HOME = "quarkus.native.graalvm-home";
    private static final String QUARKUS_CONFIG_KEY_JAVA_HOME = "quarkus.native.java-home";
    private static final String PACKAGE_NATIVE = "native";
    private static final List<String> QUARKUS_CACHEABLE_PACKAGE_TYPES = Arrays.asList("jar", "legacy-jar", "uber-jar", "native");
    private static final List<String> QUARKUS_KEYS_AS_FILE_INPUTS = Arrays.asList("quarkus.docker.dockerfile-native-path", "quarkus.docker.dockerfile-jvm-path", "quarkus.openshift.jvm-dockerfile", "quarkus.openshift.native-dockerfile");
    private static final List<String> QUARKUS_IGNORED_PROPERTIES = Arrays.asList("quarkus.native.graalvm-home", "quarkus.native.java-home");

    QuarkusBuildCache() {
    }

    void configureBuildCache(BuildCacheApi buildCache) {
        buildCache.registerNormalizationProvider(context -> {
            QuarkusExtensionConfiguration extensionConfiguration = new QuarkusExtensionConfiguration(context.getProject());
            if (extensionConfiguration.isQuarkusCacheEnabled()) {
                context.configureRuntimeClasspathNormalization(normalization -> normalization.addPropertiesNormalization(extensionConfiguration.getCurrentConfigFileName(), QUARKUS_IGNORED_PROPERTIES));
            }
        });
        buildCache.registerMojoMetadataProvider(context -> context.withPlugin("quarkus-maven-plugin", () -> {
            if ("build".equals(context.getMojoExecution().getGoal())) {
                QuarkusExtensionConfiguration extensionConfiguration = new QuarkusExtensionConfiguration(context.getProject());
                if (extensionConfiguration.isQuarkusCacheEnabled()) {
                    LOGGER.info("Configuring caching for Quarkus build");
                    this.configureQuarkusBuildGoal(context, extensionConfiguration);
                } else {
                    LOGGER.debug("Quarkus caching is disabled (gradle.quarkus.cache.enabled=false)");
                }
            }
        }));
    }

    private void configureQuarkusBuildGoal(MojoMetadataProvider.Context context, QuarkusExtensionConfiguration extensionConfiguration) {
        Properties quarkusCurrentProperties;
        String baseDir = context.getProject().getBasedir().getAbsolutePath();
        if (this.isQuarkusBuildCacheable(baseDir, extensionConfiguration, quarkusCurrentProperties = QuarkusBuildCache.loadProperties(context.getProject().getBasedir().getAbsolutePath(), extensionConfiguration.getCurrentConfigFileName()))) {
            this.configureInputs(context, extensionConfiguration, quarkusCurrentProperties);
            this.configureOutputs(context);
        } else {
            LOGGER.info("Caching not possible for Quarkus goal");
        }
    }

    private static Properties loadProperties(String baseDir, String propertyFile) {
        Properties props = new Properties();
        File configFile = new File(baseDir, propertyFile);
        if (configFile.exists()) {
            try (FileInputStream input = new FileInputStream(configFile);){
                props.load(input);
            }
            catch (IOException e) {
                LOGGER.error("Error while loading " + propertyFile, (Throwable)e);
            }
        } else {
            LOGGER.debug(propertyFile + " not found");
        }
        return props;
    }

    private boolean isQuarkusBuildCacheable(String baseDir, QuarkusExtensionConfiguration extensionConfiguration, Properties quarkusCurrentProperties) {
        return this.isPackagingTypeSupported(quarkusCurrentProperties) && this.isNotNativeOrInContainerNativeBuild(quarkusCurrentProperties) && this.isQuarkusPropertiesUnchanged(baseDir, extensionConfiguration, quarkusCurrentProperties);
    }

    private boolean isQuarkusPropertiesUnchanged(String baseDir, QuarkusExtensionConfiguration extensionConfiguration, Properties quarkusCurrentProperties) {
        Properties quarkusPreviousProperties = QuarkusBuildCache.loadProperties(baseDir, extensionConfiguration.getDumpConfigFileName());
        if (quarkusPreviousProperties.size() == 0) {
            LOGGER.debug("Quarkus previous properties not found");
            return false;
        }
        if (quarkusCurrentProperties.size() == 0) {
            LOGGER.debug("Quarkus current properties not found");
            return false;
        }
        HashSet<Map.Entry<Object, Object>> quarkusPropertiesCopy = new HashSet<Map.Entry<Object, Object>>(quarkusPreviousProperties.entrySet());
        quarkusPropertiesCopy.removeAll(quarkusCurrentProperties.entrySet());
        quarkusPropertiesCopy.removeIf(e -> QUARKUS_IGNORED_PROPERTIES.contains(e.getKey().toString()));
        if (quarkusPropertiesCopy.size() <= 0) {
            return true;
        }
        LOGGER.info("Quarkus properties have changed [" + quarkusPropertiesCopy.stream().map(e -> e.getKey().toString()).collect(Collectors.joining(", ")) + "]");
        return false;
    }

    private boolean isNotNativeOrInContainerNativeBuild(Properties quarkusCurrentProperties) {
        if (PACKAGE_NATIVE.equals(quarkusCurrentProperties.getProperty(QUARKUS_CONFIG_KEY_PACKAGE_TYPE))) {
            String builderImage = quarkusCurrentProperties.getProperty(QUARKUS_CONFIG_KEY_NATIVE_BUILDER_IMAGE, "");
            if (builderImage.isEmpty()) {
                LOGGER.info("Quarkus build is not using a fixed image");
                return false;
            }
            if (QUARKUS_CONFIG_KEY_NATIVE_CONTAINER_BUILD.stream().noneMatch(key -> Boolean.parseBoolean(quarkusCurrentProperties.getProperty((String)key)))) {
                LOGGER.info("Quarkus build is not in-container");
                return false;
            }
        }
        return true;
    }

    private boolean isPackagingTypeSupported(Properties quarkusCurrentProperties) {
        String packageType = quarkusCurrentProperties.getProperty(QUARKUS_CONFIG_KEY_PACKAGE_TYPE);
        if (packageType == null || !QUARKUS_CACHEABLE_PACKAGE_TYPES.contains(packageType)) {
            LOGGER.info("Quarkus package type " + packageType + " is not cacheable");
            return false;
        }
        return true;
    }

    private void configureInputs(MojoMetadataProvider.Context context, QuarkusExtensionConfiguration extensionConfiguration, Properties quarkusCurrentProperties) {
        context.inputs(inputs -> {
            this.addOsInputs((MojoMetadataProvider.Context.Inputs)inputs);
            this.addCompilerInputs((MojoMetadataProvider.Context.Inputs)inputs);
            this.addClasspathInput(context, (MojoMetadataProvider.Context.Inputs)inputs);
            this.addMojoInputs((MojoMetadataProvider.Context.Inputs)inputs);
            this.addQuarkusPropertiesInput((MojoMetadataProvider.Context.Inputs)inputs, extensionConfiguration);
            this.addQuarkusFilesInputs((MojoMetadataProvider.Context.Inputs)inputs, quarkusCurrentProperties);
        });
    }

    private void addOsInputs(MojoMetadataProvider.Context.Inputs inputs) {
        inputs.property("osName", (Object)System.getProperty("os.name")).property("osVersion", (Object)System.getProperty("os.version")).property("osArch", (Object)System.getProperty("os.arch"));
    }

    private void addCompilerInputs(MojoMetadataProvider.Context.Inputs inputs) {
        inputs.property("javaVersion", (Object)System.getProperty("java.version"));
    }

    private void addClasspathInput(MojoMetadataProvider.Context context, MojoMetadataProvider.Context.Inputs inputs) {
        try {
            List compileClasspathElements = context.getProject().getCompileClasspathElements();
            inputs.fileSet("quarkusCompileClasspath", (Object)compileClasspathElements, fileSet -> fileSet.normalizationStrategy(MojoMetadataProvider.Context.FileSet.NormalizationStrategy.CLASSPATH));
        }
        catch (DependencyResolutionRequiredException e) {
            throw new IllegalStateException("Classpath can't be resolved");
        }
    }

    private void addMojoInputs(MojoMetadataProvider.Context.Inputs inputs) {
        inputs.fileSet("generatedSourcesDirectory", fileSet -> {}).properties(new String[]{"appArtifact", "closeBootstrappedApp", "finalName", "ignoredEntries", "manifestEntries", "manifestSections", "skip", "skipOriginalJarRename", "systemProperties", "properties"}).ignore(new String[]{"project", "buildDir", "mojoExecution", "session", "repoSession", "repos", "pluginRepos", "attachRunnerAsMainArtifact", "bootstrapId", "buildDirectory"});
    }

    private void addQuarkusPropertiesInput(MojoMetadataProvider.Context.Inputs inputs, QuarkusExtensionConfiguration extensionConfiguration) {
        inputs.fileSet("quarkusConfigCheck", (Object)new File(extensionConfiguration.getCurrentConfigFileName()), fileSet -> fileSet.normalizationStrategy(MojoMetadataProvider.Context.FileSet.NormalizationStrategy.RELATIVE_PATH));
        inputs.fileSet("generatedSourcesDirectory", fileSet -> {});
    }

    private void addQuarkusFilesInputs(MojoMetadataProvider.Context.Inputs inputs, Properties quarkusCurrentProperties) {
        for (String quarkusFilePropertyKey : QUARKUS_KEYS_AS_FILE_INPUTS) {
            String quarkusFilePropertyValue = quarkusCurrentProperties.getProperty(quarkusFilePropertyKey);
            if (!this.isNotEmpty(quarkusFilePropertyValue)) continue;
            inputs.fileSet(quarkusFilePropertyKey, (Object)new File(quarkusFilePropertyValue), fileSet -> fileSet.normalizationStrategy(MojoMetadataProvider.Context.FileSet.NormalizationStrategy.RELATIVE_PATH));
        }
    }

    private boolean isNotEmpty(String value) {
        return value != null && !value.isEmpty();
    }

    private void configureOutputs(MojoMetadataProvider.Context context) {
        context.outputs(outputs -> {
            String quarkusExeFileName = "target/" + context.getProject().getBuild().getFinalName() + "-runner";
            String quarkusJarFileName = "target/" + context.getProject().getBuild().getFinalName() + ".jar";
            String quarkusUberJarFileName = "target/" + context.getProject().getBuild().getFinalName() + "-runner.jar";
            outputs.cacheable("this plugin has CPU-bound goals with well-defined inputs and outputs");
            outputs.file("quarkusExe", (Object)quarkusExeFileName);
            outputs.file("quarkusJar", (Object)quarkusJarFileName);
            outputs.file("quarkusUberJar", (Object)quarkusUberJarFileName);
        });
    }

    private static final class QuarkusExtensionConfiguration {
        private static final String GRADLE_QUARKUS_KEY_CACHE_ENABLED = "GRADLE_QUARKUS_CACHE_ENABLED";
        private static final String GRADLE_QUARKUS_KEY_CONFIG_FILE = "GRADLE_QUARKUS_EXTENSION_CONFIG_FILE";
        private static final String GRADLE_QUARKUS_KEY_BUILD_PROFILE = "BUILD_PROFILE";
        private static final String GRADLE_QUARKUS_DEFAULT_BUILD_PROFILE = "prod";
        private static final String GRADLE_QUARKUS_KEY_DUMP_CONFIG_PREFIX = "DUMP_CONFIG_PREFIX";
        private static final String GRADLE_QUARKUS_DEFAULT_DUMP_CONFIG_PREFIX = "quarkus";
        private static final String GRADLE_QUARKUS_KEY_DUMP_CONFIG_SUFFIX = "DUMP_CONFIG_SUFFIX";
        private static final String GRADLE_QUARKUS_DEFAULT_DUMP_CONFIG_SUFFIX = "config-dump";
        private final Properties configuration = new Properties();

        private QuarkusExtensionConfiguration(MavenProject project) {
            String isQuarkusCacheEnabledFromEnv = System.getenv(GRADLE_QUARKUS_KEY_CACHE_ENABLED);
            this.configuration.setProperty(GRADLE_QUARKUS_KEY_CACHE_ENABLED, isQuarkusCacheEnabledFromEnv != null ? isQuarkusCacheEnabledFromEnv : "");
            this.configuration.setProperty(GRADLE_QUARKUS_KEY_BUILD_PROFILE, GRADLE_QUARKUS_DEFAULT_BUILD_PROFILE);
            this.configuration.setProperty(GRADLE_QUARKUS_KEY_DUMP_CONFIG_PREFIX, GRADLE_QUARKUS_DEFAULT_DUMP_CONFIG_PREFIX);
            this.configuration.setProperty(GRADLE_QUARKUS_KEY_DUMP_CONFIG_SUFFIX, GRADLE_QUARKUS_DEFAULT_DUMP_CONFIG_SUFFIX);
            String extensionConfigurationFileFromEnv = System.getenv(GRADLE_QUARKUS_KEY_CONFIG_FILE);
            String extensionConfigurationFileFromMaven = project.getProperties().getProperty(GRADLE_QUARKUS_KEY_CONFIG_FILE.toLowerCase().replace("_", "."), "");
            if (extensionConfigurationFileFromEnv != null && !extensionConfigurationFileFromEnv.isEmpty()) {
                this.configuration.putAll((Map<?, ?>)QuarkusBuildCache.loadProperties(project.getBasedir().getAbsolutePath(), extensionConfigurationFileFromEnv));
            } else if (!extensionConfigurationFileFromMaven.isEmpty()) {
                this.configuration.putAll((Map<?, ?>)QuarkusBuildCache.loadProperties(project.getBasedir().getAbsolutePath(), extensionConfigurationFileFromMaven));
            }
        }

        private boolean isQuarkusCacheEnabled() {
            return !Boolean.FALSE.toString().equals(this.configuration.get(GRADLE_QUARKUS_KEY_CACHE_ENABLED));
        }

        private String getDumpConfigFileName() {
            return String.format(".quarkus/%s-%s-%s", this.configuration.getProperty(GRADLE_QUARKUS_KEY_DUMP_CONFIG_PREFIX), this.configuration.getProperty(GRADLE_QUARKUS_KEY_BUILD_PROFILE), this.configuration.getProperty(GRADLE_QUARKUS_KEY_DUMP_CONFIG_SUFFIX));
        }

        private String getCurrentConfigFileName() {
            return String.format("target/%s-%s-config-check", this.configuration.getProperty(GRADLE_QUARKUS_KEY_DUMP_CONFIG_PREFIX), this.configuration.getProperty(GRADLE_QUARKUS_KEY_BUILD_PROFILE));
        }
    }
}

