package com.gradle;

import org.apache.maven.project.MavenProject;

import java.util.Properties;

final class QuarkusExtensionConfiguration {

    // Environment variable key to disable caching
    private static final String DEVELOCITY_QUARKUS_KEY_CACHE_ENABLED = "DEVELOCITY_QUARKUS_CACHE_ENABLED";

    // Environment variable or Maven property key to define extension configuration file location
    private static final String DEVELOCITY_QUARKUS_KEY_CONFIG_FILE = "DEVELOCITY_QUARKUS_EXTENSION_CONFIG_FILE";

    // Extension configuration build profile key
    private static final String DEVELOCITY_QUARKUS_KEY_BUILD_PROFILE = "BUILD_PROFILE";

    // Extension configuration default profile
    private static final String DEVELOCITY_QUARKUS_DEFAULT_BUILD_PROFILE = "prod";

    // Extension configuration dump config file prefix
    private static final String DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_PREFIX = "DUMP_CONFIG_PREFIX";

    // Extension configuration default dump config file prefix
    private static final String DEVELOCITY_QUARKUS_DEFAULT_DUMP_CONFIG_PREFIX = "quarkus";

    // Extension configuration dump config file suffix
    private static final String DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_SUFFIX = "DUMP_CONFIG_SUFFIX";

    // Extension configuration default dump config file suffix
    private static final String DEVELOCITY_QUARKUS_DEFAULT_DUMP_CONFIG_SUFFIX = "config-dump";

    private final Properties configuration = new Properties();

    QuarkusExtensionConfiguration(MavenProject project) {
        // loading default properties
        String isQuarkusCacheEnabledFromEnv = System.getenv(DEVELOCITY_QUARKUS_KEY_CACHE_ENABLED);
        configuration.setProperty(DEVELOCITY_QUARKUS_KEY_CACHE_ENABLED, isQuarkusCacheEnabledFromEnv != null ? isQuarkusCacheEnabledFromEnv : "");
        configuration.setProperty(DEVELOCITY_QUARKUS_KEY_BUILD_PROFILE, DEVELOCITY_QUARKUS_DEFAULT_BUILD_PROFILE);
        configuration.setProperty(DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_PREFIX, DEVELOCITY_QUARKUS_DEFAULT_DUMP_CONFIG_PREFIX);
        configuration.setProperty(DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_SUFFIX, DEVELOCITY_QUARKUS_DEFAULT_DUMP_CONFIG_SUFFIX);

        // loading optional overridden locations
        String extensionConfigurationFileFromEnv = System.getenv(DEVELOCITY_QUARKUS_KEY_CONFIG_FILE);
        String extensionConfigurationFileFromMaven =
                project.getProperties().getProperty(
                        DEVELOCITY_QUARKUS_KEY_CONFIG_FILE.toLowerCase().replace("_", "."),
                        ""
                );

        if (extensionConfigurationFileFromEnv != null && !extensionConfigurationFileFromEnv.isEmpty()) {
            // override default properties from configuration file defined in the environment
            configuration.putAll(QuarkusExtensionUtil.loadProperties(project.getBasedir().getAbsolutePath(), extensionConfigurationFileFromEnv));
        } else if (!extensionConfigurationFileFromMaven.isEmpty()) {
            // override default properties from configuration file defined as Maven property
            configuration.putAll(QuarkusExtensionUtil.loadProperties(project.getBasedir().getAbsolutePath(), extensionConfigurationFileFromMaven));
        }
    }

    /**
     * @return whether Quarkus cache is enabled or not
     */
    boolean isQuarkusCacheEnabled() {
        // Quarkus cache is enabled by default
        return !Boolean.FALSE.toString().equals(configuration.get(DEVELOCITY_QUARKUS_KEY_CACHE_ENABLED));
    }

    /**
     * This file contains Quarkus' properties used to configure the application.
     * This file is generated by the Quarkus build goal.
     *
     * @return dump config file name
     */
    String getDumpConfigFileName() {
        return String.format(".quarkus/%s-%s-%s",
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_PREFIX),
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_BUILD_PROFILE),
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_SUFFIX)
        );
    }

    /**
     * This file contains Quarkus' properties values when process-resources phase is executed.
     * It is generated by the Quarkus track-config-changes goal.
     *
     * @return config check file name
     */
    String getCurrentConfigFileName() {
        return String.format("target/%s-%s-config-check",
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_PREFIX),
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_BUILD_PROFILE)
        );
    }

    String getCurrentDependencyFileName() {
        return String.format("target/%s-%s-dependencies.txt",
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_PREFIX),
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_BUILD_PROFILE)
        );
    }

    String getCurrentDependencyChecksumsFileName() {
        return String.format("target/%s-%s-dependency-checksums.txt",
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_DUMP_CONFIG_PREFIX),
                configuration.getProperty(DEVELOCITY_QUARKUS_KEY_BUILD_PROFILE)
        );
    }

}
