package com.greenbird.text;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.springframework.beans.BeanWrapper;
import org.springframework.beans.BeanWrapperImpl;

import java.beans.PropertyDescriptor;
import java.util.Arrays;

/**
 * toString builder that is based on the {@code ReflectionToStringBuilder} from Apache Commons Lang but
 * accesses JavaBean properties instead of using reflection on fields.
 */
public class BeanPropertyToStringBuilder extends ReflectionToStringBuilder {
    public BeanPropertyToStringBuilder(Object object) {
        super(object);
        init();
    }

    public BeanPropertyToStringBuilder(Object object, ToStringStyle style) {
        super(object, style);
        init();
    }

    private void init() {
        // we do not need to traverse the class hierarchy as the BeanWrapper does that for us.
        super.setUpToClass(getObject().getClass());
    }

    protected void appendFieldsIn(Class<?> clazz) {
        if (clazz.isArray()) {
            reflectionAppendArray(getObject());
        } else {
            BeanWrapper beanWrapper = new BeanWrapperImpl(getObject());
            for (PropertyDescriptor propertyDescriptor : beanWrapper.getPropertyDescriptors()) {
                String propertyName = propertyDescriptor.getName();
                if (acceptProperty(propertyName)) {
                    append(propertyName, beanWrapper.getPropertyValue(propertyName));
                }
            }
        }
    }

    public ReflectionToStringBuilder setExcludedPropertyNames(String... propertyNames) {
        return super.setExcludeFieldNames(propertyNames);
    }

    private boolean acceptProperty(String property) {
        // Reject fields from the getExcludeFieldNames list.
        return !("class".equals(property) || (excludeFieldNames != null && Arrays.binarySearch(excludeFieldNames, property) >= 0));
    }

    @Override
    public void setUpToClass(Class<?> clazz) {
        throw new UnsupportedOperationException("Class hierarchy limitation not supported.");
    }
}
