package com.greenbird.xml;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

/**
 * Utility that simplifies the usage of the standard JDK
 * <a href="http://download.oracle.com/javase/6/docs/api/javax/xml/xpath/XPath.html" target="_blank">XPath</a> API
 * by removing checked exceptions and adding generics.
 * <p/>
 * Note that this utility is not optimized for performance and should not be used in non-performance critical scenarios.
 */
public class XPathRoot {
    private XPath xpathParser = XPathFactory.newInstance().newXPath();
    private Node root;
    private NamespaceContext namespaceContext;

    private XPathRoot(InputSource inputSource) {
        try {
            DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
            domFactory.setNamespaceAware(true);
            DocumentBuilder builder = domFactory.newDocumentBuilder();
            root = builder.parse(inputSource);
        } catch (Exception e) {
            throw new RuntimeException("Failed to initialise xml document.", e);
        }
    }

    private XPathRoot(Node node) {
        root = node;
    }

    public static XPathRoot forNode(Node node) {
        return new XPathRoot(node);
    }

    public static XPathRoot forSource(InputSource xmlSource) {
        return new XPathRoot(xmlSource);
    }

    public static XPathRoot forString(String xmlString) {
        return forSource(new InputSource(new StringReader(xmlString)));
    }

    public static XPathRoot forResource(String xmlPath) {
        return forSource(new InputSource(Thread.currentThread().getContextClassLoader().getResourceAsStream(xmlPath)));
    }

    public XPathRoot withNamespaceContext(NamespaceContext namespaceContext) {
        this.namespaceContext = namespaceContext;
        if (namespaceContext != null) {
            xpathParser.setNamespaceContext(namespaceContext);
        }
        return this;
    }

    public List<Node> nodes(String xPath) {
        return nodes(xPath, root);
    }

    public List<Node> nodes(String xPath, Object context) {
        NodeList nodeList = evaluate(xPath, context, XPathConstants.NODESET);

        List<Node> nodes = new ArrayList<Node>();
        for (int i = 0; i < nodeList.getLength(); i++) {
            nodes.add(nodeList.item(i));
        }
        return nodes;
    }

    public Node node(String xPath) {
        return node(xPath, root);
    }

    public Node node(String xPath, Object context) {
        return evaluate(xPath, context, XPathConstants.NODE);
    }

    public String value(String xPath) {
        return evaluate(xPath, root, XPathConstants.STRING);
    }

    public String value(String xPath, Object context) {
        return evaluate(xPath, context, XPathConstants.STRING);
    }

    public XPathRoot rootFrom(String xPath) {
        return rootFrom(node(xPath));
    }

    public XPathRoot rootFrom(Node rootNode) {
        return forNode(rootNode).withNamespaceContext(namespaceContext);
    }

    @SuppressWarnings("unchecked")
    private <T> T evaluate(String xPath, Object context, QName type) {
        try {
            return (T) xpathParser.evaluate(removeAbsoluteRootPathElement(xPath), context, type);
        } catch (XPathExpressionException e) {
            throw new RuntimeException(String.format("Failed to parse expression '%s' for context %s.", xPath, context), e);
        }
    }

    // By removing the initial '/' from all queries
    // we stop absolute queries on sub-roots from failing
    private String removeAbsoluteRootPathElement(String xPath) {
        String trimmedXpath = xPath;
        if (xPath.startsWith("/") && xPath.charAt(1) != '/') {
            trimmedXpath = xPath.substring(1);
        }
        return trimmedXpath;
    }
}
