/*
 * [Licensed per the Open Source "MIT License".]
 *
 * Copyright (c) 2006 - 2017 by
 * Global Technology Consulting Group, Inc. at
 * http://gtcGroup.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.gtcgroup.justify.cdi.helper.internal;

import java.lang.annotation.Annotation;

import javax.enterprise.inject.spi.BeanManager;

import org.apache.deltaspike.cdise.api.CdiContainer;
import org.apache.deltaspike.cdise.api.CdiContainerLoader;
import org.apache.deltaspike.core.api.provider.BeanManagerProvider;
import org.apache.deltaspike.core.api.provider.BeanProvider;

import com.gtcgroup.justify.core.exception.internal.JustifyRuntimeException;
import com.gtcgroup.justify.core.helper.JstPortableJndiUtilHelper;

/**
 * This Util Helper class provides support for CDI managed bean processing.
 *
 * <p style="font-family:Verdana; font-size:10px; font-style:italic">
 * Copyright (c) 2006 - 2017 by Global Technology Consulting Group, Inc. at
 * <a href="http://gtcGroup.com">gtcGroup.com </a>.
 * </p>
 *
 * @author Marvin Toll
 * @since v3.0
 */

public enum BeanManagerUtilHelper {

	@SuppressWarnings("javadoc")
	INSTANCE;

	private static CdiContainer cdiContainer;

	@SuppressWarnings("javadoc")
	public static final String JNDI_BEANMANAGER = "java:comp/BeanManager";

	static {
		BeanManagerUtilHelper.initializeCdiBeanManager();
		BeanManagerUtilHelper.initializeJndiContainer();
	}

	private static JustifyRuntimeException createException(final Class<?> clazz, final Exception e,
			final String additionalMessage) {

		final StringBuilder message = new StringBuilder();
		message.append("The class [");
		message.append(clazz.getName());
		message.append("] could not be found.");
		message.append(additionalMessage);

		return new JustifyRuntimeException(message.toString() + "\n\t[" + e.getMessage() + "].\n");
	}

	/**
	 * @return {@link BeanManager}
	 */
	public static BeanManager getBeanManager() {
		return BeanManagerProvider.getInstance().getBeanManager();
	}

	/**
	 * @return {@link CdiContainer}
	 */
	public static CdiContainer getCdiContainer() {
		return BeanManagerUtilHelper.cdiContainer;
	}

	private static void initializeCdiBeanManager() {

		BeanManagerUtilHelper.cdiContainer = CdiContainerLoader.getCdiContainer();
		BeanManagerUtilHelper.cdiContainer.boot();
		BeanManagerUtilHelper.cdiContainer.getContextControl().startContexts();
	}

	/**
	 * @return boolean
	 */
	@SafeVarargs
	public static boolean initializeCdiScope(final Class<? extends Annotation>... scopeClasses) {

		for (final Class<? extends Annotation> clazz : scopeClasses) {

			BeanManagerUtilHelper.cdiContainer.getContextControl().stopContext(clazz);
			BeanManagerUtilHelper.cdiContainer.getContextControl().startContext(clazz);
		}
		return true;
	}

	/**
	 * @return boolean
	 */
	public static boolean initializeCdiScopes() {

		BeanManagerUtilHelper.cdiContainer.getContextControl().stopContexts();
		BeanManagerUtilHelper.cdiContainer.getContextControl().startContexts();
		return true;
	}

	private static void initializeJndiContainer() {

		final BeanManager beanManager = BeanManagerUtilHelper.getBeanManager();

		JstPortableJndiUtilHelper.bind(BeanManagerUtilHelper.JNDI_BEANMANAGER, beanManager);
	}

	/**
	 * This method performs dependency injection on an instance. It is useful
	 * for instances which are not a CDI managed bean.
	 *
	 * @return boolean
	 */
	public static boolean injectFields(final Object instance) {

		BeanProvider.injectFields(instance);
		return true;
	}

	/**
	 * @return {@link Object}
	 */
	public static <BEAN> BEAN retrieveManagedBean(final Class<BEAN> clazz) {

		return retrieveManagedBean(clazz, false);
	}

	/**
	 * @return {@link Object}
	 */
	public static <BEAN> BEAN retrieveManagedBean(final Class<BEAN> clazz, final boolean suppressException) {

		BEAN bean = null;

		try {
			bean = BeanProvider.getContextualReference(clazz, false);

		} catch (final Exception e) {

			if (suppressException) {

				bean = null;
			} else {

				throw BeanManagerUtilHelper.createException(clazz, e,
						"\n\n\t*** Ensure a beans.xml file is properly configured. ***\n");
			}
		}
		return bean;
	}
}