/*
 * [Licensed per the Open Source "MIT License".]
 *
 * Copyright (c) 2006 - 2017 by
 * Global Technology Consulting Group, Inc. at
 * http://gtcGroup.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
package com.gtcgroup.justify.cdi.rule;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;

import org.junit.Rule;
import org.junit.rules.TestRule;

import com.gtcgroup.justify.cdi.helper.internal.BeanManagerUtilHelper;
import com.gtcgroup.justify.core.base.JstBaseRule;
import com.gtcgroup.justify.core.exception.internal.RuleException;
import com.gtcgroup.justify.core.helper.internal.ArrayUtilHelper;

/**
 * This {@link Rule} class typically satisfies CDI injections on a test class
 * instance. It will also satisfy CDI injects on any unmanaged POJOs if
 * required.
 *
 * <p style="font-family:Verdana; font-size:10px; font-style:italic">
 * Copyright (c) 2006 - 2017 by Global Technology Consulting Group, Inc. at
 * <a href="http://gtcGroup.com">gtcGroup.com </a>.
 * </p>
 *
 * @author Marvin Toll
 * @since v3.0
 */
public class JstConfigureCdiRule extends JstBaseRule {

	/**
	 * @return {@link TestRule}
	 */
	@SuppressWarnings("unchecked")
    public static <RULE extends JstConfigureCdiRule> RULE withThisTestInstance(final Object thiz) {

		return (RULE) new JstConfigureCdiRule(thiz);
	}

	private Object testInstance;

	protected JstConfigureCdiRule(final Object testInstance) {

		super();

		if (null == testInstance || Boolean.class.isAssignableFrom(testInstance.getClass())) {

			throw new RuleException("\n\nThe API is not being invoked properly... "
					+ "consider entering 'this' to trigger a scan of the test class for CDI annotations.\n");
		}
		boolean found = false;

		final Field[] fields1 = testInstance.getClass().getFields();
		final Field[] fields2 = testInstance.getClass().getDeclaredFields();

		final Field[] fieldAll = ArrayUtilHelper.joinArrays(fields1, fields2);

		for (final Field field : fieldAll) {

			final Annotation annotation = field.getAnnotation(Rule.class);

			if (null != annotation) {
				found = true;
				break;
			}
		}

		if (!found) {
			System.out.println("\t*** Warning: @Rule may have been inadvertently omitted from the test ["
					+ testInstance.getClass().getSimpleName() + "] class. ***");
		}

		this.testInstance = testInstance;
		return;
	}

	/**
	 * @see JstBaseRule#afterTM()
	 */
	@Override
	public void afterTM() {

		return;
	}

	/**
	 * @see JstBaseRule#beforeTM()
	 */
	@Override
	public void beforeTM() {

		BeanManagerUtilHelper.injectFields(this.testInstance);

		BeanManagerUtilHelper.initializeCdiScopes();

		return;
	}
}