/*
 * ProGuard -- shrinking, optimization, obfuscation, and preverification
 *             of Java bytecode.
 *
 * Copyright (c) 2002-2020 Guardsquare NV
 */
package proguard.classfile.util;

import proguard.classfile.*;
import proguard.classfile.attribute.Attribute;
import proguard.classfile.attribute.CodeAttribute;
import proguard.classfile.attribute.visitor.AttributeVisitor;
import proguard.classfile.editor.InstructionSequenceBuilder;
import proguard.classfile.instruction.Instruction;
import proguard.classfile.visitor.ParameterVisitor;

import java.util.HashMap;
import java.util.Map;

import static proguard.classfile.TypeConstants.*;
import static proguard.classfile.util.ClassUtil.isInternalCategory2Type;

/**
 * A utility class to allocate free variables inside a given method.
 *
 * @author Thomas Neidhart
 */
public class VariableAllocator
implements   AttributeVisitor,
             ParameterVisitor
{
    private final boolean supportsInstructionPreloading;

    private int maxLocals        =  0;
    private int localVariable    = -1;
    private int instanceVariable = -1;

    private final InstructionSequenceBuilder builder;

    /**
     * Keeps track of the original offsets of method parameters.
     */
    private final Map<Integer, Integer> originalOffsets = new HashMap<>();

    /**
     * Keeps track of the parameter types.
     */
    private final Map<Integer, String> parameterTypes = new HashMap<>();

    /**
     * Keeps track of the parameters which have already been retrieved
     */
    private final Map<Integer, Integer> newParameterVariables = new HashMap<>();


    public VariableAllocator(ProgramClass  programClass,
                             ProgramMethod programMethod,
                             boolean       supportsInstructionPreloading)
    {
        this.supportsInstructionPreloading = supportsInstructionPreloading;

        if (programMethod != null)
        {
            programMethod.accept(programClass, new AllParameterVisitor(false, this));
            programMethod.attributesAccept(programClass, this);
        }

        builder = new InstructionSequenceBuilder(programClass);
    }


    public int getLocalVariable()
    {
        if (localVariable == -1)
        {
            localVariable = maxLocals++;
        }
        return localVariable;
    }


    public boolean supportsInstructionPreloading()
    {
        return supportsInstructionPreloading;
    }


    public int getInstanceVariable()
    {
        if (!supportsInstructionPreloading())
        {
            throw new IllegalStateException("Trying to get the instance variable but VariableAllocator can't support preloading.");
        }

        if (instanceVariable == -1)
        {
            instanceVariable = maxLocals++;

            builder.aload_0()
                   .astore(instanceVariable);
        }
        return instanceVariable;
    }


    /**
     * Creates a new local variable to store the value of the parameter provided
     *
     * @param parameterIndex the index of the parameter to consider
     * @return the variable index which holds the parameter value
     */
    public int getParameterVariable(int parameterIndex)
    {
        if (!supportsInstructionPreloading())
        {
            throw new IllegalStateException("Trying to get the parameter variable but VariableAllocator can't support preloading.");
        }

        if (!this.originalOffsets.containsKey(parameterIndex))
        {
            throw new IllegalStateException("Trying to get the parameter for index " + parameterIndex + " but it does not exists.");
        }

        String parameterType = this.parameterTypes.get(parameterIndex);
        int parameterOffset  = this.originalOffsets.get(parameterIndex);
        int newLocalOffset;

        if (newParameterVariables.containsKey(parameterOffset))
        {
            newLocalOffset = newParameterVariables.get(parameterOffset);
        }
        else
        {
            newLocalOffset = maxLocals;
            // Category2 types (long, double) take up 2 local var slots
            maxLocals += isInternalCategory2Type(parameterType) ? 2 : 1;
            newParameterVariables.put(parameterOffset, newLocalOffset);

            char type = parameterType.charAt(0);
            switch (type)
            {
                case BOOLEAN:
                case BYTE:
                case CHAR:
                case SHORT:
                case INT:
                    builder.iload(parameterOffset)
                           .istore(newLocalOffset);
                    break;
                case LONG:
                    builder.lload(parameterOffset)
                           .lstore(newLocalOffset);
                    break;
                case FLOAT:
                    builder.fload(parameterOffset)
                           .fstore(newLocalOffset);
                    break;
                case DOUBLE:
                    builder.dload(parameterOffset)
                           .dstore(newLocalOffset);
                    break;
                case CLASS_START:
                case ARRAY:
                    builder.aload(parameterOffset)
                           .astore(newLocalOffset);
                    break;
                default:
                    throw new RuntimeException("Cannot determine of parameter type with signature " + parameterType);
            }
        }

        return newLocalOffset;
    }


    public int getNextGlobalVariable()
    {
        return maxLocals++;
    }


    public int getNextWideGlobalVariable()
    {
        int result = maxLocals;
        maxLocals += 2;
        return result;
    }


    public Instruction[] preloadInstructions()
    {
        return builder.instructions();
    }


    // Implementations for AttributeVisitor.

    @Override
    public void visitAnyAttribute(Clazz clazz, Attribute attribute) {}


    @Override
    public void visitCodeAttribute(Clazz clazz, Method method, CodeAttribute codeAttribute)
    {
        maxLocals = codeAttribute.u2maxLocals;
    }


    // Implementations for ParameterVisitor

    @Override
    public void visitParameter(Clazz  clazz,
                               Member member,
                               int    parameterIndex,
                               int    parameterCount,
                               int    parameterOffset,
                               int    parameterSize,
                               String parameterType,
                               Clazz  referencedClass)
    {
        boolean isStatic = (member.getAccessFlags() & AccessConstants.STATIC) != 0;
        this.originalOffsets.put(parameterIndex, isStatic ? parameterOffset : parameterOffset + 1);
        this.parameterTypes.put(parameterIndex, parameterType);
    }
}
