/*
 * ProGuardCORE -- library to process Java bytecode.
 *
 * Copyright (c) 2002-2020 Guardsquare NV
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package proguard.classfile.constant;

import proguard.classfile.*;
import proguard.classfile.visitor.*;

/**
 * This {@link Constant} represents a ref constant in the constant pool.
 *
 * @author Eric Lafortune
 */
public abstract class RefConstant extends Constant {
  public int u2classIndex;
  public int u2nameAndTypeIndex;

  /**
   * An extra field pointing to the referenced Clazz object. This field is typically filled out by
   * the <code>{@link
   * proguard.classfile.util.ClassReferenceInitializer
   * ClassReferenceInitializer}</code>.
   */
  public Clazz referencedClass;

  /*
   * The subclasses contain the properly typed referenced
   * member objects/
   */

  /** Creates an uninitialized RefConstant. */
  protected RefConstant() {}

  /** Returns the class index. */
  public int getClassIndex() {
    return u2classIndex;
  }

  /** Returns the name-and-type index. */
  public int getNameAndTypeIndex() {
    return u2nameAndTypeIndex;
  }

  /** Sets the name-and-type index. */
  public void setNameAndTypeIndex(int index) {
    u2nameAndTypeIndex = index;
  }

  /** Returns the class name. */
  public String getClassName(Clazz clazz) {
    return clazz.getClassName(u2classIndex);
  }

  /** Returns the method/field name. */
  public String getName(Clazz clazz) {
    return clazz.getName(u2nameAndTypeIndex);
  }

  /** Returns the type. */
  public String getType(Clazz clazz) {
    return clazz.getType(u2nameAndTypeIndex);
  }

  /** Lets the referenced class accept the given visitor. */
  public void referencedClassAccept(ClassVisitor classVisitor) {
    if (referencedClass != null) {
      referencedClass.accept(classVisitor);
    }
  }

  /** Lets the referenced class member accept the given visitor. */
  public abstract void referencedMemberAccept(MemberVisitor memberVisitor);

  // Implementations for Object.

  @Override
  public boolean equals(Object object) {
    if (object == null || !this.getClass().equals(object.getClass())) {
      return false;
    }

    if (this == object) {
      return true;
    }

    RefConstant other = (RefConstant) object;

    return this.u2classIndex == other.u2classIndex
        && this.u2nameAndTypeIndex == other.u2nameAndTypeIndex;
  }

  @Override
  public int hashCode() {
    return getTag() ^ (u2classIndex << 5) ^ (u2nameAndTypeIndex << 16);
  }

  @Override
  public String toString() {
    return "Ref(" + u2classIndex + "," + u2nameAndTypeIndex + ")";
  }
}
