/*
 * Copyright (c) 2008-2013, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.queue.tx;

import com.hazelcast.nio.ObjectDataInput;
import com.hazelcast.nio.ObjectDataOutput;
import com.hazelcast.queue.QueueService;
import com.hazelcast.spi.Invocation;
import com.hazelcast.spi.NodeEngine;
import com.hazelcast.spi.Operation;
import com.hazelcast.transaction.impl.KeyAwareTransactionLog;
import com.hazelcast.util.ExceptionUtil;

import java.io.IOException;
import java.util.concurrent.Future;

/**
 * @ali 3/25/13
 */
public class QueueTransactionLog implements KeyAwareTransactionLog {

    long itemId;
    String name;
    Operation op;
    int partitionId;

    public QueueTransactionLog() {
    }

    public QueueTransactionLog(long itemId, String name, int partitionId, Operation op) {
        this.itemId = itemId;
        this.name = name;
        this.partitionId = partitionId;
        this.op = op;
    }

    public Future prepare(NodeEngine nodeEngine) {
        boolean pollOperation = op instanceof TxnPollOperation;
        TxnPrepareOperation operation = new TxnPrepareOperation(name, itemId, pollOperation);
        try {
            Invocation invocation = nodeEngine.getOperationService()
                    .createInvocationBuilder(QueueService.SERVICE_NAME, operation, partitionId).build();
            return invocation.invoke();
        } catch (Throwable t) {
            throw ExceptionUtil.rethrow(t);
        }
    }

    public Future commit(NodeEngine nodeEngine) {
        try {
            Invocation invocation = nodeEngine.getOperationService()
                    .createInvocationBuilder(QueueService.SERVICE_NAME, op, partitionId).build();
            return invocation.invoke();
        } catch (Throwable t) {
            throw ExceptionUtil.rethrow(t);
        }
    }

    public Future rollback(NodeEngine nodeEngine) {
        boolean pollOperation = op instanceof TxnPollOperation;
        TxnRollbackOperation operation = new TxnRollbackOperation(name, itemId, pollOperation);
        try {
            Invocation invocation = nodeEngine.getOperationService()
                    .createInvocationBuilder(QueueService.SERVICE_NAME, operation, partitionId).build();
            return invocation.invoke();
        } catch (Throwable t) {
            throw ExceptionUtil.rethrow(t);
        }
    }

    public void writeData(ObjectDataOutput out) throws IOException {
        out.writeLong(itemId);
        out.writeUTF(name);
        out.writeInt(partitionId);
        out.writeObject(op);
    }

    public void readData(ObjectDataInput in) throws IOException {
        itemId = in.readLong();
        name = in.readUTF();
        partitionId = in.readInt();
        op = in.readObject();
    }

    public Object getKey() {
        return itemId;
    }
}
