/*
 * Copyright (c) 2008-2019, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.cache.impl.operation;

import com.hazelcast.cache.impl.CacheDataSerializerHook;
import com.hazelcast.cache.impl.record.CacheRecord;
import com.hazelcast.core.Member;
import com.hazelcast.internal.cluster.Versions;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.ObjectDataInput;
import com.hazelcast.nio.ObjectDataOutput;
import com.hazelcast.nio.serialization.Data;
import com.hazelcast.spi.BackupAwareOperation;
import com.hazelcast.spi.Operation;
import com.hazelcast.spi.impl.MutatingOperation;
import com.hazelcast.spi.impl.operationservice.TargetAware;
import com.hazelcast.version.Version;

import javax.cache.expiry.ExpiryPolicy;
import java.io.IOException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.hazelcast.util.MapUtil.createHashMap;

public class CachePutAllOperation extends CacheOperation
        implements BackupAwareOperation, MutableOperation, MutatingOperation, TargetAware {

    private List<Map.Entry<Data, Data>> entries;
    private ExpiryPolicy expiryPolicy;
    private int completionId;

    private transient Map<Data, CacheRecord> backupRecords;
    private transient Address target;

    public CachePutAllOperation() {
    }

    public CachePutAllOperation(String cacheNameWithPrefix, List<Map.Entry<Data, Data>> entries, ExpiryPolicy expiryPolicy,
                                int completionId) {
        super(cacheNameWithPrefix);
        this.entries = entries;
        this.expiryPolicy = expiryPolicy;
        this.completionId = completionId;
    }

    @Override
    public int getCompletionId() {
        return completionId;
    }

    @Override
    public void setCompletionId(int completionId) {
        this.completionId = completionId;
    }

    @Override
    public void run() throws Exception {
        String callerUuid = getCallerUuid();
        backupRecords = createHashMap(entries.size());

        for (Map.Entry<Data, Data> entry : entries) {
            Data key = entry.getKey();
            Data value = entry.getValue();

            CacheRecord backupRecord = recordStore.put(key, value, expiryPolicy, callerUuid, completionId);

            // backupRecord may be null (eg expired on put)
            if (backupRecord != null) {
                backupRecords.put(key, backupRecord);
                publishWanUpdate(key, backupRecord);
            }
        }
    }

    @Override
    public boolean shouldBackup() {
        return !backupRecords.isEmpty();
    }

    @Override
    public Operation getBackupOperation() {
        return new CachePutAllBackupOperation(name, backupRecords);
    }

    @Override
    public int getId() {
        return CacheDataSerializerHook.PUT_ALL;
    }

    @Override
    public void setTarget(Address address) {
        this.target = address;
    }

    @Override
    protected boolean requiresExplicitServiceName() {
        // RU_COMPAT_3_10
        Member member = getNodeEngine().getClusterService().getMember(target);
        if (member == null) {
            return false;
        }
        Version memberVersion = member.getVersion().asVersion();
        return memberVersion.isLessThan(Versions.V3_11);
    }

    @Override
    protected void writeInternal(ObjectDataOutput out) throws IOException {
        super.writeInternal(out);
        out.writeObject(expiryPolicy);
        out.writeInt(completionId);
        out.writeInt(entries.size());
        for (Map.Entry<Data, Data> entry : entries) {
            out.writeData(entry.getKey());
            out.writeData(entry.getValue());
        }
    }

    @Override
    protected void readInternal(ObjectDataInput in) throws IOException {
        super.readInternal(in);
        expiryPolicy = in.readObject();
        completionId = in.readInt();
        int size = in.readInt();
        entries = new ArrayList<Map.Entry<Data, Data>>(size);
        for (int i = 0; i < size; i++) {
            Data key = in.readData();
            Data value = in.readData();
            entries.add(new AbstractMap.SimpleImmutableEntry<Data, Data>(key, value));
        }
    }

}
