/*
 * Copyright (c) 2008-2019, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.core;


import com.hazelcast.spi.annotation.Beta;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.Semaphore;

import static com.hazelcast.util.Preconditions.checkNotNull;
import static com.hazelcast.util.Preconditions.checkPositive;


/**
 * The Pipeline can be used to speed up requests. It is build on top of asynchronous
 * requests like e.g. {@link IMap#getAsync(Object)} or any other asynchronous call.
 *
 * The main purpose of the pipeline is to control the number of concurrent requests
 * when using asynchronous invocations. This can be done by setting the depth using
 * the constructor. So you could set the depth to e.g 100 and do 1000 calls. That means
 * that at any given moment, there will only be 100 concurrent requests.
 *
 * It depends on the situation what the optimal depth (number of invocations in
 * flight) should be. If it is too high, you can run into memory related problems.
 * If it is too low, it will provide little or no performance advantage at all. In
 * most cases a pipeline and a few hundred map/cache puts/gets should not lead to any
 * problems. For testing purposes we frequently have a pipeline of 1000 or more
 * concurrent requests to be able to saturate the system.
 *
 * The Pipeline can't be used for transaction purposes. So you can't create a
 * Pipeline, add a set of asynchronous request and then not call {@link #results()}
 * to prevent executing these requests. Invocations can be executed before the
 * {@link #results()} is called.
 *
 * Pipelines can be used by both clients or members.
 *
 * The Pipeline isn't threadsafe. So only a single thread should add requests to
 * the pipeline and wait for results.
 *
 * Currently all {@link ICompletableFuture} and their responses are stored in the
 * Pipeline. So be careful executing a huge number of request with a single pipeline
 * because it can lead to a huge memory bubble. In this cases it is better to
 * periodically, after waiting for completion, to replace the pipeline by a new one.
 * In the future we might provide this as an out of the box experience, but currently
 * we do not.
 *
 * A pipeline provides its own backpressure on the system. So there will not be more
 * in flight invocations than the depth of the pipeline. This means that the pipeline
 * will work fine when backpressure on the client/member is disabled (default). Also
 * when it it enabled it will work fine, but keep in mind that the number of concurrent
 * invocations in the pipeline could be lower than the configured number of invocation
 * of the pipeline because the backpressure on the client/member is leading.
 *
 * The Pipeline has been marked as Beta since we need to see how the API needs to
 * evolve. But there is no problem using it in production. We use similar techniques
 * to achieve high performance.
 *
 * @param <E>
 */
@Beta
public class Pipeline<E> {

    // we are using a caller runs executor to prevent having the overhead of creating a task and kicking an executor.
    // the only thing that gets executed in future callback is the return of the license to the semaphore.
    private static final Executor EXECUTOR = new Executor() {
        @Override
        public void execute(Runnable command) {
            command.run();
        }
    };

    private final Semaphore semaphore;
    private final List<ICompletableFuture<E>> futures = new ArrayList<ICompletableFuture<E>>();

    /**
     * Creates a pipeline with the given depth.
     *
     * @param depth the maximum number of concurrent calls allowed in this pipeline.
     * @throws IllegalArgumentException if depth smaller than 1. But if you use depth 1, it means that
     *                                  every call is sync and you will not benefit from pipelining at all.
     */
    public Pipeline(int depth) {
        checkPositive(depth, "depth must be positive");
        this.semaphore = new Semaphore(depth);
    }

    /**
     * Returns the results.
     * <p>
     * The results are returned in the order the requests were done.
     * <p>
     * This call waits till all requests have completed.
     *
     * @return the List of results.
     * @throws Exception is something fails getting the results.
     */
    public List<E> results() throws Exception {
        List<E> result = new ArrayList<E>(futures.size());
        for (ICompletableFuture<E> f : futures) {
            result.add(f.get());
        }
        return result;
    }

    /**
     * Adds a future to this Pipeline or blocks until there is capacity to add the future to the pipeline.
     * <p>
     * This call blocks until there is space in the pipeline, but it doesn't mean that the invocation that
     * returned the ICompletableFuture got blocked.
     *
     * @param future the future to add.
     * @return the future added.
     * @throws InterruptedException if the Thread got interrupted while adding the request to the pipeline.
     * @throws NullPointerException if future is null.
     */
    public ICompletableFuture<E> add(ICompletableFuture<E> future) throws InterruptedException {
        checkNotNull(future, "future can't be null");

        semaphore.acquire();
        futures.add(future);
        future.andThen(new ExecutionCallback<E>() {
            @Override
            public void onResponse(E response) {
                semaphore.release();
            }

            @Override
            public void onFailure(Throwable t) {
                semaphore.release();
            }
        }, EXECUTOR);
        return future;
    }
}
