/*
 * Copyright (c) 2008-2019, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.nio.tcp;

import com.hazelcast.instance.EndpointQualifier;
import com.hazelcast.instance.ProtocolType;
import com.hazelcast.internal.metrics.MetricsRegistry;
import com.hazelcast.internal.metrics.Probe;
import com.hazelcast.internal.networking.ChannelInitializerProvider;
import com.hazelcast.logging.LoggingService;
import com.hazelcast.nio.IOService;
import com.hazelcast.nio.NetworkingService;
import com.hazelcast.spi.properties.HazelcastProperties;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import static com.hazelcast.internal.metrics.ProbeLevel.MANDATORY;
import static com.hazelcast.nio.ConnectionType.MEMCACHE_CLIENT;
import static com.hazelcast.nio.ConnectionType.REST_CLIENT;

class TcpIpUnifiedEndpointManager
        extends TcpIpEndpointManager {

    TcpIpUnifiedEndpointManager(NetworkingService root, EndpointQualifier qualifier,
                                ChannelInitializerProvider channelInitializerProvider,
                                IOService ioService, LoggingService loggingService, MetricsRegistry metricsRegistry,
                                HazelcastProperties properties) {
        super(root, qualifier, channelInitializerProvider, ioService, loggingService,
                metricsRegistry, properties, ProtocolType.valuesAsSet());
    }

    Set<TcpIpConnection> getTextConnections() {
        Set<TcpIpConnection> connections = activeConnections.isEmpty()
                ? Collections.<TcpIpConnection>emptySet()
                : new HashSet<TcpIpConnection>(activeConnections.size());

        for (TcpIpConnection conn : activeConnections) {
            if (conn.isAlive() && conn.getType() == REST_CLIENT || conn.getType() == MEMCACHE_CLIENT) {
                connections.add(conn);
            }
        }
        return connections;
    }

    Set<TcpIpConnection> getCurrentClientConnections() {
        Set<TcpIpConnection> connections = activeConnections.isEmpty()
                ? Collections.<TcpIpConnection>emptySet()
                : new HashSet<TcpIpConnection>(activeConnections.size());

        for (TcpIpConnection conn : activeConnections) {
            if (conn.isAlive() && conn.isClient()) {
                connections.add(conn);
            }
        }
        return connections;
    }

    //TODO (TK) Cache the values

    @Probe(name = "clientCount", level = MANDATORY)
    public int getCurrentClientConnectionsCount() {
        return getCurrentClientConnections().size();
    }

    @Probe(name = "textCount", level = MANDATORY)
    public int getCurrentTextConnections() {
        return getTextConnections().size();
    }
}
