/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.nio;

import com.hazelcast.logging.Logger;
import com.hazelcast.util.QuickMath;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.UTFDataFormatException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;

public final class UTFEncoderDecoder {
    private static final int STRING_CHUNK_SIZE = 16384;
    private static final UTFEncoderDecoder INSTANCE = UTFEncoderDecoder.buildUTFUtil();
    private final StringCreator stringCreator;
    private final boolean hazelcastEnterpriseActive;

    private UTFEncoderDecoder(boolean fastStringCreator) {
        this(fastStringCreator ? UTFEncoderDecoder.buildFastStringCreator() : new DefaultStringCreator(), false);
    }

    private UTFEncoderDecoder(StringCreator stringCreator, boolean hazelcastEnterpriseActive) {
        this.stringCreator = stringCreator;
        this.hazelcastEnterpriseActive = hazelcastEnterpriseActive;
    }

    public StringCreator getStringCreator() {
        return this.stringCreator;
    }

    public static void writeUTF(DataOutput out, String str, byte[] buffer) throws IOException {
        INSTANCE.writeUTF0(out, str, buffer);
    }

    public static String readUTF(DataInput in, byte[] buffer) throws IOException {
        return INSTANCE.readUTF0(in, buffer);
    }

    public boolean isHazelcastEnterpriseActive() {
        return this.hazelcastEnterpriseActive;
    }

    public void writeUTF0(DataOutput out, String str, byte[] buffer) throws IOException {
        if (!QuickMath.isPowerOfTwo(buffer.length)) {
            throw new IllegalArgumentException("Size of the buffer has to be power of two, was " + buffer.length);
        }
        boolean isNull = str == null;
        out.writeBoolean(isNull);
        if (isNull) {
            return;
        }
        int length = str.length();
        out.writeInt(length);
        if (length > 0) {
            int chunkSize = length / 16384 + 1;
            for (int i = 0; i < chunkSize; ++i) {
                int beginIndex = Math.max(0, i * 16384 - 1);
                int endIndex = Math.min((i + 1) * 16384 - 1, length);
                this.writeShortUTF(out, str, beginIndex, endIndex, buffer);
            }
        }
    }

    private void writeShortUTF(DataOutput out, String str, int beginIndex, int endIndex, byte[] buffer) throws IOException {
        int i;
        int utfLength = this.calculateUtf8Length(str, beginIndex, endIndex);
        if (utfLength > 65535) {
            throw new UTFDataFormatException("encoded string too long:" + utfLength + " bytes");
        }
        out.writeShort(utfLength);
        char c = '\u0000';
        int count = 0;
        for (i = beginIndex; i < endIndex && (c = str.charAt(i)) >= '\u0001' && c <= '\u007f'; ++i) {
            this.buffering(buffer, count++, (byte)c, out);
        }
        while (i < endIndex) {
            c = str.charAt(i);
            if (c >= '\u0001' && c <= '\u007f') {
                this.buffering(buffer, count++, (byte)c, out);
            } else if (c > '\u07ff') {
                this.buffering(buffer, count++, (byte)(0xE0 | c >> 12 & 0xF), out);
                this.buffering(buffer, count++, (byte)(0x80 | c >> 6 & 0x3F), out);
                this.buffering(buffer, count++, (byte)(0x80 | c & 0x3F), out);
            } else {
                this.buffering(buffer, count++, (byte)(0xC0 | c >> 6 & 0x1F), out);
                this.buffering(buffer, count++, (byte)(0x80 | c & 0x3F), out);
            }
            ++i;
        }
        int length = count % buffer.length;
        out.write(buffer, 0, length == 0 ? buffer.length : length);
    }

    public String readUTF0(DataInput in, byte[] buffer) throws IOException {
        if (!QuickMath.isPowerOfTwo(buffer.length)) {
            throw new IllegalArgumentException("Size of the buffer has to be power of two, was " + buffer.length);
        }
        boolean isNull = in.readBoolean();
        if (isNull) {
            return null;
        }
        int length = in.readInt();
        char[] data = new char[length];
        if (length > 0) {
            int chunkSize = length / 16384 + 1;
            for (int i = 0; i < chunkSize; ++i) {
                int beginIndex = Math.max(0, i * 16384 - 1);
                this.readShortUTF(in, data, beginIndex, buffer);
            }
        }
        return this.stringCreator.buildString(data);
    }

    private void readShortUTF(DataInput in, char[] data, int beginIndex, byte[] buffer) throws IOException {
        int utflen = in.readShort();
        int count = 0;
        int charArrCount = beginIndex;
        while (count < utflen) {
            int c = this.buffered(buffer, count++, utflen, in) & 0xFF;
            count = this.decodeChar(in, data, buffer, utflen, count, charArrCount, c);
            ++charArrCount;
        }
    }

    private int decodeChar(DataInput in, char[] data, byte[] buffer, int utflen, int count, int charArrCount, int c) throws IOException {
        switch (c >> 4) {
            case 0: 
            case 1: 
            case 2: 
            case 3: 
            case 4: 
            case 5: 
            case 6: 
            case 7: {
                this.decodeOneByteChar(data, charArrCount, c);
                return count;
            }
            case 12: 
            case 13: {
                return this.decodeTwoBytesChar(data, charArrCount, c, in, buffer, utflen, count);
            }
            case 14: {
                return this.decodeThreeBytesChar(data, charArrCount, c, in, buffer, utflen, count);
            }
        }
        throw new UTFDataFormatException("malformed input around byte " + count);
    }

    private int decodeThreeBytesChar(char[] data, int charArrCount, int char1, DataInput in, byte[] buffer, int utflen, int count) throws IOException {
        int pos = count;
        if (count + 2 > utflen) {
            throw new UTFDataFormatException("malformed input: partial character at end");
        }
        byte char2 = this.buffered(buffer, pos++, utflen, in);
        byte char3 = this.buffered(buffer, pos++, utflen, in);
        if ((char2 & 0xC0) != 128 || (char3 & 0xC0) != 128) {
            throw new UTFDataFormatException("malformed input around byte " + (count - 1));
        }
        int c1 = (char1 & 0xF) << 12;
        int c2 = (char2 & 0x3F) << 6;
        int c3 = (char3 & 0x3F) << 0;
        data[charArrCount] = (char)(c1 | c2 | c3);
        return pos;
    }

    private int decodeTwoBytesChar(char[] data, int charArrCount, int char1, DataInput in, byte[] buffer, int utflen, int count) throws IOException {
        byte char2;
        int pos = count;
        if (count + 1 > utflen) {
            throw new UTFDataFormatException("malformed input: partial character at end");
        }
        if (((char2 = this.buffered(buffer, pos++, utflen, in)) & 0xC0) != 128) {
            throw new UTFDataFormatException("malformed input around byte " + count);
        }
        data[charArrCount] = (char)((char1 & 0x1F) << 6 | char2 & 0x3F);
        return pos;
    }

    private void decodeOneByteChar(char[] data, int charArrCount, int c) {
        data[charArrCount] = (char)c;
    }

    private int calculateUtf8Length(String value, int beginIndex, int endIndex) {
        int utfLength = 0;
        for (int i = beginIndex; i < endIndex; ++i) {
            char c = value.charAt(i);
            if (c >= '\u0001' && c <= '\u007f') {
                ++utfLength;
                continue;
            }
            if (c > '\u07ff') {
                utfLength += 3;
                continue;
            }
            utfLength += 2;
        }
        return utfLength;
    }

    private void buffering(byte[] buffer, int pos, byte value, DataOutput out) throws IOException {
        int innerPos = QuickMath.mod(pos, buffer.length);
        if (pos > 0 && innerPos == 0) {
            out.write(buffer, 0, buffer.length);
        }
        buffer[innerPos] = value;
    }

    private byte buffered(byte[] buffer, int pos, int utfLength, DataInput in) throws IOException {
        int innerPos = QuickMath.mod(pos, buffer.length);
        if (innerPos == 0) {
            int length = Math.min(buffer.length, utfLength - pos);
            in.readFully(buffer, 0, length);
        }
        return buffer[innerPos];
    }

    public static boolean useOldStringConstructor() {
        try {
            Class<String> clazz = String.class;
            clazz.getDeclaredConstructor(Integer.TYPE, Integer.TYPE, char[].class);
            return true;
        }
        catch (Throwable t) {
            Logger.getLogger(UTFEncoderDecoder.class).finest("Old String constructor doesn't seem available", t);
            return false;
        }
    }

    private static UTFEncoderDecoder buildUTFUtil() {
        try {
            Class<?> clazz = Class.forName("com.hazelcast.nio.utf8.EnterpriseStringCreator");
            Method method = clazz.getDeclaredMethod("findBestStringCreator", new Class[0]);
            return new UTFEncoderDecoder((StringCreator)method.invoke(clazz, new Object[0]), true);
        }
        catch (Throwable t) {
            Logger.getLogger(UTFEncoderDecoder.class).finest("EnterpriseStringCreator not available on classpath", t);
            boolean faststringEnabled = Boolean.parseBoolean(System.getProperty("hazelcast.nio.faststring", "true"));
            return new UTFEncoderDecoder(faststringEnabled ? UTFEncoderDecoder.buildFastStringCreator() : new DefaultStringCreator(), false);
        }
    }

    private static StringCreator buildFastStringCreator() {
        try {
            Constructor<String> constructor = null;
            constructor = UTFEncoderDecoder.useOldStringConstructor() ? String.class.getDeclaredConstructor(Integer.TYPE, Integer.TYPE, char[].class) : String.class.getDeclaredConstructor(char[].class, Boolean.TYPE);
            if (constructor != null) {
                constructor.setAccessible(true);
                return new FastStringCreator(constructor);
            }
        }
        catch (Throwable t) {
            Logger.getLogger(UTFEncoderDecoder.class).finest("No fast string creator seems to available, falling back to reflection", t);
        }
        return null;
    }

    public static interface StringCreator {
        public String buildString(char[] var1);
    }

    private static class FastStringCreator
    implements StringCreator {
        private final Constructor<String> constructor;
        private final boolean useOldStringConstructor;

        public FastStringCreator(Constructor<String> constructor) {
            this.constructor = constructor;
            this.useOldStringConstructor = constructor.getParameterTypes().length == 3;
        }

        @Override
        public String buildString(char[] chars) {
            try {
                if (this.useOldStringConstructor) {
                    return this.constructor.newInstance(0, chars.length, chars);
                }
                return this.constructor.newInstance(chars, Boolean.TRUE);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    private static class DefaultStringCreator
    implements StringCreator {
        private DefaultStringCreator() {
        }

        @Override
        public String buildString(char[] chars) {
            return new String(chars);
        }
    }
}

