/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.map.impl.eviction;

import com.hazelcast.config.EvictionPolicy;
import com.hazelcast.config.MapConfig;
import com.hazelcast.config.MaxSizeConfig;
import com.hazelcast.core.EntryEventType;
import com.hazelcast.map.impl.MapEventPublisher;
import com.hazelcast.map.impl.MapServiceContext;
import com.hazelcast.map.impl.RecordStore;
import com.hazelcast.map.impl.eviction.MaxSizeChecker;
import com.hazelcast.map.impl.record.Record;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.serialization.Data;
import com.hazelcast.spi.EventService;
import com.hazelcast.spi.NodeEngine;
import com.hazelcast.util.MemoryInfoAccessor;
import java.util.Arrays;
import java.util.Iterator;

public final class EvictionOperator {
    private static final int ONE_HUNDRED_PERCENT = 100;
    private MapServiceContext mapServiceContext;
    private MaxSizeChecker maxSizeChecker;

    private EvictionOperator() {
    }

    public static EvictionOperator create(MapServiceContext mapServiceContext) {
        EvictionOperator evictionOperator = new EvictionOperator();
        MaxSizeChecker maxSizeChecker = new MaxSizeChecker(mapServiceContext);
        evictionOperator.setMaxSizeChecker(maxSizeChecker);
        evictionOperator.setMapServiceContext(mapServiceContext);
        return evictionOperator;
    }

    public static EvictionOperator create(MemoryInfoAccessor memoryInfoAccessor, MapServiceContext mapServiceContext) {
        EvictionOperator evictionOperator = new EvictionOperator();
        MaxSizeChecker maxSizeChecker = new MaxSizeChecker(memoryInfoAccessor, mapServiceContext);
        evictionOperator.setMaxSizeChecker(maxSizeChecker);
        evictionOperator.setMapServiceContext(mapServiceContext);
        return evictionOperator;
    }

    public void setMapServiceContext(MapServiceContext mapServiceContext) {
        this.mapServiceContext = mapServiceContext;
    }

    public void setMaxSizeChecker(MaxSizeChecker maxSizeChecker) {
        this.maxSizeChecker = maxSizeChecker;
    }

    public MaxSizeChecker getMaxSizeChecker() {
        return this.maxSizeChecker;
    }

    public void removeEvictableRecords(RecordStore recordStore, int evictableSize, MapConfig mapConfig, boolean backup) {
        MapServiceContext mapServiceContext = this.mapServiceContext;
        EvictionPolicy evictionPolicy = mapConfig.getEvictionPolicy();
        long[] criterias = this.createAndPopulateEvictionCriteriaArray(recordStore, evictionPolicy);
        if (criterias == null) {
            return;
        }
        Arrays.sort(criterias);
        int evictableBaseIndex = this.getEvictionStartIndex(criterias, evictableSize);
        long criteriaValue = criterias[evictableBaseIndex];
        int evictedRecordCounter = 0;
        Iterator<Record> iterator = recordStore.iterator();
        while (iterator.hasNext()) {
            Record record = iterator.next();
            long value = this.getEvictionCriteriaValue(record, evictionPolicy);
            if (value <= criteriaValue) {
                Data tmpKey = record.getKey();
                Object tmpValue = record.getValue();
                if (this.evictIfNotLocked(tmpKey, recordStore, backup)) {
                    ++evictedRecordCounter;
                    String mapName = mapConfig.getName();
                    if (!backup) {
                        mapServiceContext.interceptAfterRemove(mapName, value);
                        this.fireEvent(tmpKey, tmpValue, mapName, mapServiceContext);
                    }
                }
            }
            if (evictedRecordCounter < evictableSize) continue;
            break;
        }
    }

    private long[] createAndPopulateEvictionCriteriaArray(RecordStore recordStore, EvictionPolicy evictionPolicy) {
        int size = recordStore.size();
        long[] criterias = null;
        int index = 0;
        Iterator<Record> iterator = recordStore.iterator();
        while (iterator.hasNext()) {
            Record record = iterator.next();
            if (criterias == null) {
                criterias = new long[size];
            }
            criterias[index] = this.getEvictionCriteriaValue(record, evictionPolicy);
            if (++index != size) continue;
            break;
        }
        if (criterias == null) {
            return null;
        }
        if (index < criterias.length) {
            for (int i = index; i < criterias.length; ++i) {
                criterias[i] = Long.MAX_VALUE;
            }
        }
        return criterias;
    }

    private int getEvictionStartIndex(long[] criterias, int evictableSize) {
        int length = criterias.length;
        int sizeToEvict = Math.min(evictableSize, length);
        int index = sizeToEvict - 1;
        return index < 0 ? 0 : index;
    }

    public void fireEvent(Data key, Object value, String mapName, MapServiceContext mapServiceContext) {
        if (!this.hasListener(mapName)) {
            return;
        }
        MapEventPublisher mapEventPublisher = mapServiceContext.getMapEventPublisher();
        NodeEngine nodeEngine = mapServiceContext.getNodeEngine();
        Address thisAddress = nodeEngine.getThisAddress();
        Data dataValue = mapServiceContext.toData(value);
        mapEventPublisher.publishEvent(thisAddress, mapName, EntryEventType.EVICTED, true, key, dataValue, null);
    }

    private boolean hasListener(String mapName) {
        String serviceName = this.mapServiceContext.serviceName();
        EventService eventService = this.mapServiceContext.getNodeEngine().getEventService();
        return eventService.hasEventRegistration(serviceName, mapName);
    }

    private boolean evictIfNotLocked(Data key, RecordStore recordStore, boolean backup) {
        if (recordStore.isLocked(key)) {
            return false;
        }
        recordStore.evict(key, backup);
        return true;
    }

    public int evictableSize(int currentPartitionSize, MapConfig mapConfig) {
        int evictableSize;
        int maxSize = mapConfig.getMaxSizeConfig().getSize();
        MaxSizeConfig.MaxSizePolicy maxSizePolicy = mapConfig.getMaxSizeConfig().getMaxSizePolicy();
        int evictionPercentage = mapConfig.getEvictionPercentage();
        switch (maxSizePolicy) {
            case PER_PARTITION: {
                int targetSizePerPartition = Double.valueOf((double)maxSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case PER_NODE: {
                int memberCount = this.mapServiceContext.getNodeEngine().getClusterService().getMembers().size();
                int maxPartitionSize = maxSize * memberCount / this.mapServiceContext.getNodeEngine().getPartitionService().getPartitionCount();
                int targetSizePerPartition = Double.valueOf((double)maxPartitionSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case USED_HEAP_PERCENTAGE: 
            case USED_HEAP_SIZE: 
            case FREE_HEAP_PERCENTAGE: 
            case FREE_HEAP_SIZE: {
                evictableSize = Math.max(currentPartitionSize * evictionPercentage / 100, 1);
                break;
            }
            default: {
                throw new IllegalArgumentException("Max size policy is not defined [" + (Object)((Object)maxSizePolicy) + "]");
            }
        }
        return evictableSize;
    }

    private long getEvictionCriteriaValue(Record record, EvictionPolicy evictionPolicy) {
        long value;
        switch (evictionPolicy) {
            case LRU: 
            case LFU: {
                value = record.getEvictionCriteriaNumber();
                break;
            }
            default: {
                throw new IllegalArgumentException("Not an appropriate eviction policy [" + (Object)((Object)evictionPolicy) + ']');
            }
        }
        return value;
    }
}

