/*
 * Copyright (c) 2008-2015, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.monitor.impl;

import com.hazelcast.com.hazelcast.com.hazelcast.com.hazelcast.com.hazelcast.com.eclipsesource.json.JsonObject;
import com.hazelcast.monitor.LocalTopicStats;
import com.hazelcast.util.Clock;

import java.util.concurrent.atomic.AtomicLongFieldUpdater;

import static com.hazelcast.util.JsonUtil.getLong;

public class LocalTopicStatsImpl
        implements LocalTopicStats {

    private static final AtomicLongFieldUpdater<LocalTopicStatsImpl> TOTAL_PUBLISHES_UPDATER = AtomicLongFieldUpdater
            .newUpdater(LocalTopicStatsImpl.class, "totalPublishes");
    private static final AtomicLongFieldUpdater<LocalTopicStatsImpl> TOTAL_RECEIVED_MESSAGES_UPDATER = AtomicLongFieldUpdater
            .newUpdater(LocalTopicStatsImpl.class, "totalReceivedMessages");
    private long creationTime;

    // These fields are only accessed through the updaters
    private volatile long totalPublishes;
    private volatile long totalReceivedMessages;

    public LocalTopicStatsImpl() {
        creationTime = Clock.currentTimeMillis();
    }

    @Override
    public long getCreationTime() {
        return creationTime;
    }

    @Override
    public long getPublishOperationCount() {
        return totalPublishes;
    }

    public void incrementPublishes() {
        TOTAL_PUBLISHES_UPDATER.incrementAndGet(this);
    }

    @Override
    public long getReceiveOperationCount() {
        return totalReceivedMessages;
    }

    public void incrementReceives() {
        TOTAL_RECEIVED_MESSAGES_UPDATER.incrementAndGet(this);
    }

    @Override
    public JsonObject toJson() {
        JsonObject root = new JsonObject();
        root.add("creationTime", creationTime);
        root.add("totalPublishes", totalPublishes);
        root.add("totalReceivedMessages", totalReceivedMessages);
        return root;
    }

    @Override
    public void fromJson(JsonObject json) {
        creationTime = getLong(json, "creationTime", -1L);
        TOTAL_PUBLISHES_UPDATER.set(this, getLong(json, "totalPublishes", -1L));
        TOTAL_RECEIVED_MESSAGES_UPDATER.set(this, getLong(json, "totalReceivedMessages", -1L));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        LocalTopicStatsImpl that = (LocalTopicStatsImpl) o;

        if (creationTime != that.creationTime) {
            return false;
        }
        if (totalPublishes != that.totalPublishes) {
            return false;
        }
        if (totalReceivedMessages != that.totalReceivedMessages) {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        int result = (int) (creationTime ^ (creationTime >>> 32));
        result = 31 * result + (int) (totalPublishes ^ (totalPublishes >>> 32));
        result = 31 * result + (int) (totalReceivedMessages ^ (totalReceivedMessages >>> 32));
        return result;
    }

    @Override
    public String toString() {
        return "LocalTopicStatsImpl{"
                + "creationTime=" + creationTime
                + ", totalPublishes=" + totalPublishes
                + ", totalReceivedMessages=" + totalReceivedMessages
                + '}';
    }
}
